// TODO: https://st.yandex-team.ru/CRM-10202
import * as React from 'react';
import Promise from 'bluebird';
import cx from 'classnames';
import { connect } from 'react-redux';
import Icon2 from '@crm/components/dist/lego2/Icon';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import Loadable from 'utils/Loadable';
import applyMixins from 'utils/applyMixins';
import Access from 'utils/Access';
import { ETypeString } from 'types/entities';
import { modal, AppModuleString } from 'modules/requestForm2';
import { get } from 'api/common';
import { UpsaleFactorsComponent } from 'components/UpsaleFactors';
import Mail from 'components/Mail';
import Icon from 'lego/components/Icon';
import { Categorization, CategorizationValue } from '../Categorization';
import { MailPreview } from './MailPreview';
import * as TYPES from '../types';
import css from './Mail.module.css';
import Item, { Action } from '../Item';
import { allModals } from '../../Forms';
import TogglePreviewButton from '../TogglePreviewButton';
import { mapColor } from './Mail.utils';
import { DIRECTION_ICON } from './Mail.constants';
import { Header } from './Header';

interface State {
  isLoad: boolean;
  isOpen: boolean;
  // TODO: add types for mail
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  data: any;
  loadDate: string;
}

interface ConnectedState {
  mail: TYPES.Mail;
  access: TYPES.MailProps;
  categorization?: CategorizationValue;
}

interface ConnectedDispatch {
  onDelete: () => void;
  onCreateST: (values: unknown) => unknown;
  onDetach: (values: unknown) => unknown;
  onPrintAttachments: () => void;
}

interface OwnProps {
  id: number;
  nodeId: string;
  issueId: number;
  timelineId?: string;
  redux: {
    name: string;
  };
  access: TYPES.MailProps;
}

interface Props {
  id: number;
  mail: TYPES.Mail;
  issueId: number;
  nodeId: string;
  timelineId?: string;
  redux: {
    name: string;
  };
  className?: string;
  size?: string;
  isOpen?: boolean;
  onReply: (mailId: number, method: string) => void;
}

class MailIssue extends React.Component<Props & ConnectedState & ConnectedDispatch, State> {
  public constructor(props: Props & ConnectedState & ConnectedDispatch) {
    super(props);
    this.state = {
      isLoad: false,
      isOpen: Boolean(props.isOpen),
      loadDate: '',
      data: {},
    };
    this.mailLoad = this.mailLoad.bind(this);
    this.categorizationPreview = this.categorizationPreview.bind(this);
  }

  public componentDidMount(): void {
    if (this.props.access.isOpen !== 0 || this.props.isOpen) {
      this.load();
    }
  }

  public componentDidUpdate(): void {
    if (this.state.isLoad && this.state.isOpen && this.state.loadDate !== this.props.mail.dt) {
      this.load();
    }
  }

  public componentWillUnmount(): void {
    this.loadCancel();
  }

  private handleClick = (): void => {
    if (this.state.isLoad) {
      const { isOpen } = this.state;

      this.setState({ isOpen: !isOpen });

      return;
    }

    this.load();
  };

  public mailLoad() {
    const { mail } = this.props;

    return get({
      url: '/view/mail/preview',
      data: {
        mailId: mail.id,
      },
      global: false,
    });
  }

  public loadRequest(): Promise<unknown> {
    return this.mailLoad().then((data) => {
      if (data) {
        this.setState({ isLoad: true, isOpen: true, data, loadDate: data.date });
      }
    });
  }

  private categorizationPreview() {
    const {
      mail,
      className,
      onPrintAttachments,
      onReply,
      onCreateST,
      onDetach,
      issueId,
      access,
      categorization,
      ...other
    } = this.props;

    return <MailPreview mail={mail} issueId={issueId} load={this.mailLoad} {...other} />;
  }

  public render(): React.ReactElement {
    const {
      mail,
      className,
      onPrintAttachments,
      onReply,
      onCreateST,
      onDetach,
      issueId,
      access,
      categorization,
      ...other
    } = this.props;
    const { type, bodyPreview, dt, author, hasAttach, id, isExternal, upsaleFactors } = mail;
    const { isOpen, data } = this.state;

    let actions: Action[] = [];

    if (type === TYPES.MailType.Draft && Access.isEdit(access.access)) {
      actions = [
        {
          key: 'draft',
          onClick: () => onReply(mail.id, 'draft'),
          children: 'Редактировать',
        },
        {
          key: 'delete',
          onClick: this.props.onDelete,
          children: 'Удалить',
        },
      ];
    } else {
      if (Access.isEdit(access.canReply)) {
        actions.push({
          key: 'reply',
          onClick: () => onReply(mail.id, 'reply'),
          children: 'Ответить',
        });
      }

      if (Access.isEdit(access.canReplyAll)) {
        actions.push({
          key: 'replyAll',
          onClick: () => onReply(mail.id, 'replyAll'),
          children: 'Ответить всем',
        });
      }

      if (Access.isEdit(access.canForward)) {
        actions.push({
          key: 'forward',
          onClick: () => onReply(mail.id, 'forward'),
          children: 'Переслать',
        });
      }

      if (Access.isEdit(access.canForwardInternal)) {
        actions.push({
          key: 'forwardInternal',
          onClick: () => onReply(mail.id, 'forwardInternal'),
          children: 'Смежникам',
          title: 'Переслать смежникам',
        });
      }

      if (Access.isEdit(access.canPrintAttachments)) {
        actions.push({
          key: 'printAttachments',
          onClick: onPrintAttachments,
          children: 'Напечатать все вложения',
        });
      }

      if (Access.isEdit(access.canDetach)) {
        actions.push({
          key: 'detach',
          onClick: () =>
            allModals.detachTicket.open({
              mailId: id,
              issueId,
              onSubmit: onDetach,
              workplace: 'detachMail',
              eId: issueId,
              eType: ETypeString.Issue,
            }),
          icon: (cls) => <Icon2 className={cls} svg="split" />,
          title: 'Отделить',
        });
      }

      if (Access.isRead(access.access)) {
        actions.push({
          key: 'downloadMail',
          type: 'link',
          url: `${window.CRM_SPACE_API_HOST}/download/mail/${this.props.id}/${this.props.id}.msg`,
          icon: (cls) => <Icon2 className={cls} svg="download" />,
          title: 'Скачать письмо',
        });
      }

      if (Access.isEdit(access.canCreateTicketST)) {
        actions.push({
          key: 'startrek',
          onClick: () =>
            allModals.createStartrek.open({ mailId: id, issueId, onSubmit: onCreateST }),
          icon: (cls) => <Icon2 className={cls} svg="startrek-color" />,
          title: 'Создать тикет Ст',
        });
      }

      if (Access.isEdit(access.canCreateTicket)) {
        actions.push({
          key: 'request',
          onClick: () =>
            modal.open({
              data: {
                data: {
                  mailId: id,
                  issueId,
                  byEObject: { eid: issueId, etype: ETypeString.Issue },
                },
                // TODO: https://st.yandex-team.ru/CRM-12496
                context: { module: AppModuleString.Ticket, action: 'createTicket2LineAction' },
              },
            }),
          icon: (cls) => <Icon2 className={cls} svg="bookmark" />,
          title: 'Создать тикет на 2 линию',
        });
      }
    }

    const color = mapColor({
      isExternal,
      type,
    });

    const mailDate = new Date(mail.dt);
    const targetDescription = `Письмо от ${mailDate.toLocaleDateString(
      'ru',
    )}, ${mailDate.toLocaleTimeString('ru', {
      hour: '2-digit',
      minute: '2-digit',
    })}`;

    return (
      <Item
        className={cx(css.b__mailPreview, className)}
        author={author}
        date={dt}
        actions={actions}
        color={color}
        icon={<Icon pack={Icon.PACKS.MATERIAL} icon="mail" />}
        side={
          <>
            {DIRECTION_ICON[type]}
            {hasAttach && <Icon pack={Icon.PACKS.MATERIAL} icon="attach_file" />}
          </>
        }
        afterAuthorName={<TogglePreviewButton isOpen={isOpen} toggleAction={this.handleClick} />}
        issueId={issueId}
        {...other}
      >
        <Categorization
          issueId={issueId}
          name={other.nodeId}
          workplace={ETypeString.Mail}
          access={access.category}
          isRequireMarkup={access.isRequireMarkup}
          preview={this.categorizationPreview}
          eType={mail.eType}
          id={other.id}
          initialValue={categorization}
          targetDescription={targetDescription}
        />
        {isOpen ? (
          <Mail
            {...data}
            tags={mail.tags}
            className={css.b__mail}
            key={this.state.loadDate}
            header={this.props.redux.name === 'opportunities' ? Header : undefined}
          />
        ) : (
          <div>
            {bodyPreview}
            <span
              role="button"
              tabIndex={0}
              className={css['b__mail_read-more']}
              onClick={this.handleClick}
            >
              ...
            </span>
          </div>
        )}
        <UpsaleFactorsComponent factors={upsaleFactors} />
      </Item>
    );
  }
}

// eslint-disable-next-line @typescript-eslint/no-empty-interface
interface MailIssue extends Loadable {}
applyMixins(MailIssue, [Loadable]);

export default withReduxContext(
  connect<
    ConnectedState,
    ConnectedDispatch,
    Omit<OwnProps, 'access'> & Partial<Pick<OwnProps, 'access'>> & InjectedReduxContextProps
  >(
    (state, props) => {
      const data = props.redux.selectors.storage.mails.getItem(state, props.id) || {};

      return {
        mail: data.data,
        access: props.access || data.props,
        categorization: data.categorization,
      };
    },
    (dispatch, props) => ({
      onDelete: () => {
        if (window.confirm('Вы уверены, что хотите удалить это письмо?')) {
          dispatch(
            props.redux.slices.timelineSlice.asyncActions.mailDelete({
              mailId: props.id,
              issueId: props.issueId,
            }),
          );
        }
      },
      onCreateST: (values) => {
        return dispatch(
          props.redux.slices.issueSlice.asyncActions.fetchMutationIssue(props.issueId, {
            name: 'createStartrek',
            value: values,
          }),
        );
      },
      onDetach: (values) => {
        return dispatch(
          props.redux.slices.issueSlice.asyncActions.fetchMutationIssue(props.issueId, {
            name: 'detach',
            mailId: props.id,
            value: values,
          }),
        );
      },
      onPrintAttachments: () => {
        return dispatch(
          props.redux.slices.timelineSlice.asyncActions.printAttachments({
            mailId: props.id,
          }),
        );
      },
    }),
  )(MailIssue),
);
