// TODO: https://st.yandex-team.ru/CRM-10202
import React, { useCallback, useMemo, useContext } from 'react';
import get from 'lodash/get';
import { NewMailForm } from 'components/NewMailForm/NewMailForm.bundle/loadable';
import { NewMailFormApi } from 'components/NewMailForm/NewMailForm.types';
import filterValuesForBackend from 'components/NewMailForm/utils/filterValuesForBackend';
import { connect } from 'react-redux';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import Access from 'utils/Access';
import { MergeRef } from 'utils/mergeRef';
import { actionsMailFileUpdateLinks } from 'modules/upsale/actions';
import createEmailInput from '../EmailInputContainer';
import { loadMail } from '../api';
import { LoadMailData } from '../types';
import { ResolutionSerializer } from '../ResolutionSerializer';
import { MailFileInputContext } from '../../FileInputContext';
import { createOpportunitiesFields } from '../OpportunitiesContainer';

interface Props {
  issueId: number;
  loadMailData: LoadMailData;
  onChangePristine: (pristine: boolean) => void;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  [key: string]: any;
}

const formRef = new MergeRef<NewMailFormApi>();

const MailForm = (props: Props & InjectedReduxContextProps) => {
  const {
    issueId,
    redux,
    resolutionItems,
    onSubmit,
    loadMailData,
    onChangePristine,
    accountId,
    account,
  } = props;

  const load = useCallback(() => loadMail({ issueId, location: redux.name, ...loadMailData }), [
    issueId,
    loadMailData,
    redux.name,
  ]);
  const memoEmailInput = useMemo(() => (accountId ? createEmailInput(accountId) : undefined), [
    accountId,
  ]);
  const memoOpportunitiesInput = useMemo(
    () =>
      createOpportunitiesFields({
        account,
        issueId,
        hasNoLinkCheckbox: false,
        mailId: loadMailData.mailId || loadMailData.replyToMailId,
      }),
    [accountId, issueId, loadMailData],
  );

  const mailFileInputRef = useContext(MailFileInputContext);

  return (
    <NewMailForm
      onSubmit={onSubmit}
      onLoad={load}
      EmailInput={memoEmailInput}
      OpportunitiesInput={memoOpportunitiesInput}
      customSubmitPlaceholder="Отправить и закрыть"
      customSubmitOptions={resolutionItems}
      submitView={resolutionItems && resolutionItems.length ? 'default' : 'action'}
      objId={loadMailData.mailId}
      type={loadMailData.type}
      onChangePristine={onChangePristine}
      formApiRef={formRef}
      fileInputRef={mailFileInputRef}
      hasAutoSave
      autoSaveOnUnmount
      isPreventUnload
      preventChangeLocation={false}
    />
  );
};

export default withReduxContext(
  connect<{}, {}, Props & InjectedReduxContextProps>(
    (state, props) => {
      const item = props.redux.selectors.storage.issues.getItem(state, props.issueId);
      const account = item.data.account;
      const accountId = account?.id;
      const resolution = item.props.possibleMailCloseResolution;
      const mailSendAndClose = props.redux.selectors.storage.issues.getFieldAccess(
        state,
        props.issueId,
        'mailSendAndClose',
      );

      if (!Access.isEdit(mailSendAndClose)) {
        return {
          resolutionItems: undefined,
          account,
          accountId,
        };
      }

      if (!resolution || !resolution.items) {
        return {
          resolutionItems: [],
          account,
          accountId,
        };
      }

      return {
        resolutionItems: resolution.items.map((item) => ({
          value: ResolutionSerializer.format(item.val),
          content: item.text,
        })),
        account,
        accountId,
      };
    },
    (dispatch, props) => ({
      onSubmit: (values) => {
        const action = values.action || '';
        const isSend = !(action === 'draft' || values.action === 'autoSave');

        const handle = isSend
          ? props.redux.slices.timelineSlice.asyncActions.mailSend
          : props.redux.slices.timelineSlice.asyncActions.mailSave;

        let formattedValues = filterValuesForBackend(values);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        let requestExtends: any = {
          contextExtend: {
            main: {
              issue: {
                id: props.issueId,
              },
            },
          },
        };

        if (ResolutionSerializer.isResolution(action)) {
          requestExtends = {
            moveToClosed: true,
            resolutionId: ResolutionSerializer.parse(action),
          };
        }

        return dispatch(handle(formattedValues, requestExtends)).then((data) => {
          const nextInitialMailData = { mailId: get(data, 'data.id') };

          if (data.success === false) {
            return nextInitialMailData;
          }

          if (values.action !== 'autoSave') {
            props.onSubmitSuccess();
          }

          return nextInitialMailData;
        });
      },
      clearContainer: (args) => {
        return dispatch(actionsMailFileUpdateLinks(null, args.containerId, args.fileIdsToRemove));
      },
    }),
  )(MailForm),
);
