// TODO: https://st.yandex-team.ru/CRM-10202
import * as React from 'react';
import { connect } from 'react-redux';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import Access from 'utils/Access';
import Spinner from 'components/Spinner';
import Communication from './Communication';
import TimelineNode from './TimelineNode';
import { TimelineXivaUpdateSpy } from './TimelineXivaUpdateSpy';
import SuspenseHack from './SuspenseHack';
import css from './Timeline.scss';
import config from './config';
import * as TYPES from './types';

interface Props {
  items: unknown;
  issueId: number;
  timelineId?: string;
  showCommentForm?: boolean;
  showMailForm?: boolean;
  showMessageForm?: boolean;
  showInternalMessageForm?: boolean;
  canCommentSendAndOpen?: boolean;
  get: () => void;
  sort: string;
  isLoad: boolean;
  isFetch: boolean;
  maxAccess?: number;
  expandItems?: boolean;
  comType?: string;
}

class Timeline extends React.Component<Props> {
  private ref = React.createRef<Communication>();

  private isTimelineMounted: boolean = false;

  public componentDidMount() {
    this.props.get();
  }

  public componentDidUpdate(prevProps) {
    if (this.props.issueId && this.props.issueId !== prevProps.issueId) {
      this.props.get();
    }

    if (!this.props.isFetch && this.props.isLoad && !this.isTimelineMounted) {
      this.scrollToTimelineEl();
    }
  }

  private scrollToTimelineEl = () => {
    setTimeout(() => {
      const anchor = document.getElementById(this.props.timelineId || '');
      if (!anchor) {
        return;
      }
      this.isTimelineMounted = true;
      anchor.scrollIntoView();
    }, 0);
  };

  private handleEditComment = (comment: TYPES.Comment) => {
    if (this.ref.current) {
      this.ref.current.editComment(comment);
    }
  };

  private handleEditMessage = (message: TYPES.ChatMessage) => {
    if (this.ref.current) {
      this.ref.current.editMessage(message);
    }
  };

  private handleEditInternalMessage = (internalMessage: TYPES.ChatMessage) => {
    if (this.ref.current) {
      this.ref.current.editInternalMessage(internalMessage);
    }
  };

  private handleReplyByComment = (text: string) => {
    if (this.ref.current) {
      this.ref.current.replyByComment(text);
    }
  };

  private handleReplyByMail = (mailId: number, type: string) => {
    if (this.ref.current) {
      this.ref.current.replyByMail(mailId, type);
    }
  };

  private onSuspenseHackMount = () => {
    this.isTimelineMounted = true;
  };

  public render() {
    const {
      items,
      issueId,
      timelineId,
      showCommentForm,
      showMailForm,
      showMessageForm,
      showInternalMessageForm,
      canCommentSendAndOpen,
      sort,
      isLoad,
      isFetch,
      maxAccess,
      expandItems,
      comType,
    } = this.props;

    if (!isLoad) {
      if (isFetch) {
        return <Spinner visible modal={false} />;
      }

      return null;
    }

    const commentForm = (showCommentForm ||
      showMailForm ||
      showMessageForm ||
      showInternalMessageForm) && (
      <Communication
        key="communication"
        issueId={issueId}
        ref={this.ref}
        className={css.b__form}
        showCommentForm={showCommentForm}
        showMailForm={showMailForm}
        showMessageForm={showMessageForm}
        showInternalMessageForm={showInternalMessageForm}
        canCommentSendAndOpen={canCommentSendAndOpen}
        comType={comType}
      />
    );

    return (
      <div data-testid="timeline">
        {sort === 'newer_first' && commentForm}
        <React.Suspense
          fallback={
            <SuspenseHack
              componentDidMount={this.onSuspenseHackMount}
              componentWillUnmount={this.scrollToTimelineEl}
            />
          }
        >
          <div>
            <TimelineXivaUpdateSpy issueId={issueId} />
            {config.sortMapper[sort || 'newer_first'](items).map((id) => (
              <TimelineNode
                key={id}
                maxAccess={maxAccess}
                className={css.b__item}
                issueId={issueId}
                timelineId={timelineId}
                nodeId={id}
                editComment={showCommentForm && this.handleEditComment}
                editMessage={showMessageForm && this.handleEditMessage}
                editInternalMessage={showInternalMessageForm && this.handleEditInternalMessage}
                replyByComment={showCommentForm && this.handleReplyByComment}
                replyByMail={showMailForm && this.handleReplyByMail}
                expandItems={expandItems}
              />
            ))}
          </div>
        </React.Suspense>
        {sort === 'older_first' && commentForm}
      </div>
    );
  }
}

interface ConnectedState {
  showCommentForm: boolean;
  showMailForm: boolean;
  showMessageForm: boolean;
  canCommentSendAndOpen: boolean;
  items: unknown[];
  sort: string;
  isLoad: boolean;
  isFetch: boolean;
}

interface ConnectedDispatch {
  get: () => void;
}

interface ConnectedProps {
  issueId: number;
  maxAccess?: number;
  timelineId?: string;
  expandItems?: boolean;
  comType?: string;
  forceUseFeatureIssueTimelineV2?: boolean;
}

export default withReduxContext<ConnectedProps>(
  connect<ConnectedState, ConnectedDispatch, ConnectedProps & InjectedReduxContextProps>(
    (state, props) => {
      const { redux } = props;

      const timeline = redux.selectors.meta.timelines(state, props.issueId) || {};
      const node = redux.selectors.storage.nodes.getItem(
        state,
        `issues.${props.issueId}.timeline`,
        0,
      ) as { items: unknown[] } | undefined;
      let showCommentFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'comments') || 0;
      let showMailFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'mails') || 0;
      let showMessageFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'chats') || 0;
      let showInternalMessageFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'internalChats') || 0;
      const canCommentSendAndOpen =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'сommentSendAndOpen') ||
        0;

      if (Number.isInteger(props.maxAccess as number)) {
        showCommentFormAccess = Math.min(props.maxAccess as number, showCommentFormAccess);
        showMailFormAccess = Math.min(props.maxAccess as number, showMailFormAccess);
        showMessageFormAccess = Math.min(props.maxAccess as number, showMessageFormAccess);
        showInternalMessageFormAccess = Math.min(
          props.maxAccess as number,
          showInternalMessageFormAccess,
        );
      }

      return {
        showCommentForm: Access.isEdit(showCommentFormAccess),
        showMailForm: Access.isEdit(showMailFormAccess),
        showMessageForm: Access.isEdit(showMessageFormAccess),
        showInternalMessageForm: Access.isEdit(showInternalMessageFormAccess),
        canCommentSendAndOpen: Access.isEdit(canCommentSendAndOpen),
        items: node?.items || [],
        sort: redux.selectors.timeline.sort(state),
        isLoad: timeline.isLoad,
        isFetch: timeline.isFetch,
      };
    },
    (dispatch, props) => ({
      // load all timeline with big length === 100000, CRM-8140
      get: () =>
        dispatch(
          props.redux.slices.timelineSlice.asyncActions.fetchTimeline({
            issueId: props.issueId,
            length: 100000,
          }),
        ),
    }),
  )(Timeline),
);
