// TODO: https://st.yandex-team.ru/CRM-10202

import React from 'react';
import { AsyncTransition } from 'components/AsyncTransition';
import { FloatContainer } from 'components/FloatContainer';
import RadioButton from '@crm/components/dist/lego2/RadioButton';
import { IRadioButtonOption } from '@yandex-lego/components/RadioButton';
import Icon from '@crm/components/dist/lego2/Icon';
import { config } from 'services/Config';
import { baseCallApi } from 'api/common';
import createI18N from '@yandex-int/i18n';
import { MailForm } from '../Mail';
import CommentForm from '../CommentForm';
import MessageForm from '../MessageForm';
import { InternalMessageForm } from '../InternalMessageForm';
import css from './Communication.module.css';
import {
  CommunicationProps,
  CommunicationState,
  CommunicationTypes,
  InputRef,
} from './Communication.types';
import * as TYPES from '../types';
import {
  CONFIRM_MESSAGE,
  TYPE_MAPPER,
  COMTYPE_MAPPER,
  MAX_LENGTH,
} from './Communication.constants';
import * as keyset from './Communication.i18n';

const i18n = createI18N(keyset);
const i18nMaxLengthError = i18n('maxLengthError');
const i18nSymbols = i18n('symbols');

class Communication extends React.Component<CommunicationProps, CommunicationState> {
  static communicationTypeToIcon = {
    [CommunicationTypes.Comment]: <Icon svg="comment" svgUseSelfWidth />,
    [CommunicationTypes.Mail]: <Icon svg="mailOutline" svgUseSelfWidth />,
    [CommunicationTypes.Message]: <Icon svg="chat" svgUseSelfWidth />,
    [CommunicationTypes.InternalMessage]: <Icon svg="internalChat" svgUseSelfWidth />,
  };

  constructor(props) {
    super(props);

    let defaultTab = COMTYPE_MAPPER[props.comType];
    const availableCommunicationChannels = this.getAvailableCommunicationChannels();
    if (!defaultTab || !availableCommunicationChannels.includes(defaultTab)) {
      defaultTab = availableCommunicationChannels[0];
    }

    this.state = {
      hash: 0,
      isForceMountNextState: true,
      totalSubTasks: defaultTab === CommunicationTypes.Mail ? 2 : 0,
      tab: defaultTab,
      loadMailData: {},
      pristine: true,
      showForm: true,
      comment: undefined,
      message: undefined,
      internalMessage: undefined,
      containerId: undefined,
    };
  }

  private ref = React.createRef<InputRef>();
  private containerRef = React.createRef<HTMLDivElement>();
  private toggle: React.ReactNode = undefined;

  private isForceMountNextTab(nextTab: CommunicationTypes) {
    return (
      nextTab !== CommunicationTypes.Mail ||
      (this.state.tab === CommunicationTypes.Mail && nextTab === CommunicationTypes.Mail)
    );
  }

  private handleChangeTab = (value: unknown) => {
    if (!this.canChangeTab()) {
      return;
    }

    this.resetCommunicationState();
    this.setState({
      tab: value as CommunicationTypes,
      isForceMountNextState: this.isForceMountNextTab(value as CommunicationTypes),
      totalSubTasks: (value as CommunicationTypes) === CommunicationTypes.Mail ? 2 : 0,
    });
  };

  private resetMailCommunicationState = () => {
    this.setState({
      isForceMountNextState: true,
    });

    this.resetCommunicationState();
  };

  private resetCommunicationState = () => {
    this.setState((state) => ({
      hash: state.hash + 1,
      pristine: true,
      loadMailData: {},
      comment: undefined,
      message: undefined,
      internalMessage: undefined,
    }));
  };

  private handleClearEditComment = () => {
    this.setState((state) => ({
      hash: state.hash + 1,
      comment: undefined,
    }));
  };

  private handleClearEditMessage = () => {
    this.setState((state) => ({
      hash: state.hash + 1,
      message: undefined,
    }));
  };

  private handleClearEditInternalMessage = () => {
    this.setState((state) => ({
      hash: state.hash + 1,
      internalMessage: undefined,
    }));
  };

  private canChangeTab() {
    if (this.state.tab === CommunicationTypes.Mail || this.state.pristine) {
      return true;
    }

    return window.confirm(CONFIRM_MESSAGE);
  }

  private handleChangePristine = (pristine: boolean) => {
    this.setState({ pristine });
  };

  public editComment(comment: TYPES.Comment): void {
    if (!this.canChangeTab()) {
      return;
    }

    this.setState((state) => ({
      hash: state.hash + 1,
      tab: CommunicationTypes.Comment,
      totalSubTasks: 0,
      isForceMountNextState: true,
      pristine: true,
      comment,
      showForm: true,
    }));
  }

  public editMessage(message: TYPES.ChatMessage): void {
    if (!this.canChangeTab()) {
      return;
    }

    baseCallApi({
      url: `/issue/timeline/message/${message.id}/editcontainer`,
      global: false,
    }).then((data) => {
      this.setState((state) => ({
        hash: state.hash + 1,
        tab: CommunicationTypes.Message,
        totalSubTasks: 0,
        isForceMountNextState: true,
        pristine: true,
        message: config.value.features.newEditingFiles
          ? {
              ...message,
              files: data.files,
            }
          : message,
        showForm: true,
        containerId: data.containerId,
      }));
    });
  }

  public editInternalMessage(internalMessage: TYPES.ChatMessage): void {
    if (!this.canChangeTab()) {
      return;
    }

    if (config.value.features.newEditingFiles) {
      baseCallApi({
        url: `/issue/timeline/internalChatMessage/${internalMessage.id}/editcontainer`,
        global: false,
      }).then((data) => {
        this.setState((state) => ({
          hash: state.hash + 1,
          tab: CommunicationTypes.InternalMessage,
          totalSubTasks: 0,
          isForceMountNextState: true,
          pristine: true,
          internalMessage: {
            ...internalMessage,
            files: data.files,
          },
          showForm: true,
          containerId: data.containerId,
        }));
      });
    } else {
      this.setState((state) => ({
        hash: state.hash + 1,
        tab: CommunicationTypes.InternalMessage,
        totalSubTasks: 0,
        isForceMountNextState: true,
        pristine: true,
        internalMessage,
        showForm: true,
      }));
    }
  }

  public replyByComment(text): void {
    if (this.state.tab === CommunicationTypes.Comment) {
      if (this.ref.current) {
        this.ref.current.reply(text);
      }
    }

    if (!this.canChangeTab()) {
      return;
    }

    this.setState((state) => ({
      hash: state.hash + 1,
      tab: CommunicationTypes.Comment,
      totalSubTasks: 0,
      isForceMountNextState: true,
      pristine: true,
      comment: {
        text: `<[${text}]>\n`,
      } as TYPES.Comment,
      showForm: true,
    }));
  }

  public replyByMail(mailId: number, type: string): void {
    if (!this.canChangeTab()) {
      return;
    }

    this.setState((state) => ({
      hash: state.hash + 1,
      totalSubTasks: 2,
      isForceMountNextState: this.isForceMountNextTab(CommunicationTypes.Mail),
      tab: CommunicationTypes.Mail,
      loadMailData: TYPE_MAPPER[type](mailId),
      pristine: true,
      showForm: true,
    }));
  }

  private renderFormSwitcher() {
    const { tab } = this.state;

    const options = this.getAvailableCommunicationChannelsOptions();

    if (options.length <= 1) {
      return null;
    }

    return (
      <RadioButton
        isEventValue={false}
        size="s"
        onChange={this.handleChangeTab}
        options={options}
        value={tab!}
        width="auto"
      />
    );
  }

  private getAvailableCommunicationChannelsOptions(): IRadioButtonOption[] {
    return this.getAvailableCommunicationChannels().map((type) => ({
      value: type,
      children: Communication.communicationTypeToIcon[type],
    }));
  }

  private getAvailableCommunicationChannels(): CommunicationTypes[] {
    const { showCommentForm, showMailForm, showMessageForm, showInternalMessageForm } = this.props;

    const options: CommunicationTypes[] = [];
    if (showMessageForm) {
      options.push(CommunicationTypes.Message);
    }

    if (showMailForm) {
      options.push(CommunicationTypes.Mail);
    }

    if (showCommentForm) {
      options.push(CommunicationTypes.Comment);
    }

    if (showInternalMessageForm) {
      options.push(CommunicationTypes.InternalMessage);
    }

    return options;
  }

  private handleToggleForm = () => {
    this.setState({ showForm: !this.state.showForm });
  };

  private handleTransitionError = (prevState: CommunicationState) => {
    this.setState(prevState);
  };

  private validateTextLength = (values) => {
    const error: { [key: string]: string } = {};

    if (values?.text && values.text.length > MAX_LENGTH) {
      error._error = `${i18nMaxLengthError} ${MAX_LENGTH} ${i18nSymbols}`; // eslint-disable-line no-underscore-dangle
    }

    return error;
  };

  public render() {
    const { className } = this.props;
    const { showForm } = this.state;

    if (this.getAvailableCommunicationChannels().length === 0) {
      return null;
    }

    return (
      <FloatContainer
        show={showForm}
        className={className}
        classNameContent={css.Communication}
        innerRef={this.containerRef}
        onToggle={this.handleToggleForm}
      >
        {({ toggle }) => {
          this.toggle = toggle;
          return (
            <span data-testid="timeline-input-form" style={{ display: 'contents' }}>
              {this.renderContent()}
            </span>
          );
        }}
      </FloatContainer>
    );
  }

  private renderContent() {
    const { showCommentForm, showMailForm, showMessageForm, showInternalMessageForm } = this.props;

    return (
      <AsyncTransition state={this.state} onTransitionError={this.handleTransitionError}>
        {(state) => (
          <>
            {showCommentForm &&
              state.tab === CommunicationTypes.Comment &&
              this.renderComment(state)}
            {showMailForm && state.tab === CommunicationTypes.Mail && this.renderMail(state)}
            {showMessageForm &&
              state.tab === CommunicationTypes.Message &&
              this.renderMessage(state)}
            {showInternalMessageForm &&
              state.tab === CommunicationTypes.InternalMessage &&
              this.renderInternalMessage(state)}
          </>
        )}
      </AsyncTransition>
    );
  }

  private renderComment(state: CommunicationState) {
    const { issueId, canCommentSendAndOpen, forceUseFeatureIssueTimelineV2 } = this.props;

    const { comment } = state;

    return (
      <CommentForm
        key={comment?.id || 'new_comment'}
        issueId={issueId}
        refTextComponent={this.ref}
        className={css.b__comment}
        onSuccess={this.resetCommunicationState}
        useLocationChangeAutoSave
        isPreventUnload
        onChangePristine={this.handleChangePristine}
        focusOnMount
        addonToolbarBefore={this.renderFormSwitcher()}
        addonToolbarRightAfter={this.toggle}
        view="clear"
        actionsPosition="right"
        forceUseFeatureIssueTimelineV2={forceUseFeatureIssueTimelineV2}
        {...(comment
          ? {
              objId: comment.id,
              filesObjectName: 'ocomment',
              cancelButton: Boolean(comment.id),
              editInfoText: comment.id ? comment.text : undefined,
              initialValues: {
                text: comment.text,
                files: comment.files,
                invitees: comment.invitees,
              },
              okButtonLabel: 'Сохранить',
              cancelButtonLabel: 'Отменить',
              onClearEditClick: this.handleClearEditComment,
              onCancelClick: this.resetCommunicationState,
            }
          : {
              showSendAndOpenButton: canCommentSendAndOpen,
            })}
      />
    );
  }

  private renderMail(state: CommunicationState) {
    const { issueId } = this.props;

    const { loadMailData, hash } = state;

    return (
      <MailForm
        key={hash}
        classNameBody={css.b__body}
        issueId={issueId}
        onSubmitSuccess={this.resetMailCommunicationState}
        loadMailData={loadMailData}
        onChangePristine={this.handleChangePristine}
        addonToolbarBefore={this.renderFormSwitcher()}
        addonToolbarAfter={this.toggle}
        saveDraftChildren="Сохранить"
        popupScope={this.containerRef}
      />
    );
  }

  private renderMessage(state: CommunicationState) {
    const { issueId, forceUseFeatureIssueTimelineV2 } = this.props;

    const { message, containerId } = state;

    return (
      <MessageForm
        key={message?.id || 'new_message'}
        issueId={issueId}
        placeholder="Написать в чат"
        refTextComponent={this.ref}
        className={css.b__comment}
        onSuccess={this.resetCommunicationState}
        useLocationChangeAutoSave
        isPreventUnload
        onChangePristine={this.handleChangePristine}
        focusOnMount
        addonToolbarBefore={this.renderFormSwitcher()}
        addonToolbarRightAfter={this.toggle}
        actionsPosition="right"
        view="clear"
        forceUseFeatureIssueTimelineV2={forceUseFeatureIssueTimelineV2}
        {...(message
          ? {
              objId: message.id,
              containerId,
              filesObjectName: 'chatmessage',
              cancelButton: true,
              editInfoText: message.text,
              initialValues: {
                text: message.text,
                files: message.files,
              },
              okButtonLabel: 'Сохранить',
              cancelButtonLabel: 'Отменить',
              onClearEditClick: this.handleClearEditMessage,
              onCancelClick: this.resetCommunicationState,
            }
          : {
              validate: this.validateTextLength,
            })}
      />
    );
  }

  private renderInternalMessage(state: CommunicationState) {
    const { issueId, forceUseFeatureIssueTimelineV2 } = this.props;

    const { internalMessage, containerId } = state;

    return (
      <InternalMessageForm
        key={internalMessage?.id || 'new_internalMessage'}
        issueId={issueId}
        placeholder="Написать во внутренний чат"
        refTextComponent={this.ref}
        className={css.b__comment}
        onSuccess={this.resetCommunicationState}
        useLocationChangeAutoSave
        isPreventUnload
        onChangePristine={this.handleChangePristine}
        focusOnMount
        addonToolbarBefore={this.renderFormSwitcher()}
        addonToolbarRightAfter={this.toggle}
        actionsPosition="right"
        view="clear"
        forceUseFeatureIssueTimelineV2={forceUseFeatureIssueTimelineV2}
        {...(internalMessage
          ? {
              objId: internalMessage.id,
              containerId,
              filesObjectName: 'internalchatmessage',
              cancelButton: true,
              editInfoText: internalMessage.text,
              initialValues: {
                text: internalMessage.text,
                files: internalMessage.files,
              },
              okButtonLabel: 'Сохранить',
              cancelButtonLabel: 'Отменить',
              onClearEditClick: this.handleClearEditInternalMessage,
              onCancelClick: this.resetCommunicationState,
            }
          : {})}
      />
    );
  }
}

export default Communication;
