import { ForwardRefExoticComponent, PropsWithoutRef, RefAttributes } from 'react';
import { connect } from 'react-redux';
import { withReduxContextWithForwardRef, InjectedReduxContextProps } from 'modules/issues/redux';
import Access from 'utils/Access';
import Communication from './Communication';
import { CommunicationForm } from '../TimelineV2/types/CommunicationForm';

export interface ConnectedState {
  showCommentForm: boolean;
  showMailForm: boolean;
  canCommentSendAndOpen: boolean;
}

export interface ConnectedProps {
  issueId: number;
  maxAccess?: number;
  className?: string;
  forceUseFeatureIssueTimelineV2?: boolean;
  showCommentForm?: boolean;
  showMailForm?: boolean;
  showMessageForm?: boolean;
  showInternalMessageForm?: boolean;
}

export const TimelineCommunicationForm = withReduxContextWithForwardRef<ConnectedProps>(
  connect<ConnectedState, {}, InjectedReduxContextProps & ConnectedProps>(
    (state, props) => {
      const {
        redux,
        showCommentForm,
        showMailForm,
        showMessageForm,
        showInternalMessageForm,
      } = props;

      let showCommentFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'comments') || 0;
      let showMailFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'mails') || 0;
      let showMessageFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'chats') || 0;
      let showInternalMessageFormAccess =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'internalChats') || 0;
      const canCommentSendAndOpen =
        redux.selectors.storage.issues.getFieldAccess(state, props.issueId, 'сommentSendAndOpen') ||
        0;

      if (Number.isInteger(props.maxAccess as number)) {
        showCommentFormAccess = Math.min(props.maxAccess as number, showCommentFormAccess);
        showMailFormAccess = Math.min(props.maxAccess as number, showMailFormAccess);
        showMessageFormAccess = Math.min(props.maxAccess as number, showMessageFormAccess);
        showInternalMessageFormAccess = Math.min(
          props.maxAccess as number,
          showInternalMessageFormAccess,
        );
      }

      const comType = redux.selectors.storage.issues.getField(
        state,
        props.issueId,
        'communicationTypeId',
      );

      return {
        showCommentForm: showCommentForm ?? Access.isEdit(showCommentFormAccess),
        showMailForm: showMailForm ?? Access.isEdit(showMailFormAccess),
        showMessageForm: showMessageForm ?? Access.isEdit(showMessageFormAccess),
        showInternalMessageForm:
          showInternalMessageForm ?? Access.isEdit(showInternalMessageFormAccess),
        canCommentSendAndOpen: Access.isEdit(canCommentSendAndOpen),
        comType: (comType || { name: undefined }).name,
      };
    },
    null,
    null,
    { forwardRef: true },
  )(Communication),
) as ForwardRefExoticComponent<PropsWithoutRef<ConnectedProps> & RefAttributes<CommunicationForm>>;
