import * as React from 'react';
import Icon2 from '@crm/components/dist/lego2/Icon';
import { MailType } from 'types/Mail';
import { jsonApiCall } from 'api/common';
import { ETypeString } from 'types/entities';
import { AppModuleString, modal } from 'modules/requestForm2';
import { isEditAccess, isReadAccess } from '../utils/validateAccess';
import { Action } from '../../Timeline/Item';
import { allModals } from '../../Forms';
import { CreateActionsOptions } from '../types/ActionItemScheme';
import { VirtualListTimelineItemMail } from '../TimelineV2.types';

export const createMailActions = (
  options: CreateActionsOptions<VirtualListTimelineItemMail>,
): Action[] => {
  const { data, issueId, communicationFormRef, maxAccess } = options;
  const type = data.data.data.type;
  const mailId = data.eid;
  const access = data.data.props;

  const actions: Action[] = [];

  if (type === MailType.Draft && isEditAccess(access.access, maxAccess)) {
    actions.push(
      {
        key: 'draft',
        onClick: () => communicationFormRef.current?.replyByMail(mailId, 'draft'),
        children: 'Редактировать',
      },
      {
        key: 'delete',
        onClick: () => {
          if (window.confirm('Вы уверены, что хотите удалить это письмо?')) {
            jsonApiCall({
              url: '/actions/issue/mail/delete',
              data: {
                data: {
                  mailId,
                  issueId,
                },
                context: {
                  main: {
                    issue: {
                      id: issueId,
                    },
                  },
                },
              },
            });
          }
        },
        children: 'Удалить',
      },
    );
  } else {
    if (isEditAccess(access.canReply, maxAccess)) {
      actions.push({
        key: 'reply',
        onClick: () => communicationFormRef.current?.replyByMail(mailId, 'reply'),
        children: 'Ответить',
      });
    }

    if (isEditAccess(access.canReplyAll, maxAccess)) {
      actions.push({
        key: 'replyAll',
        onClick: () => communicationFormRef.current?.replyByMail(mailId, 'replyAll'),
        children: 'Ответить всем',
      });
    }

    if (isEditAccess(access.canForward, maxAccess)) {
      actions.push({
        key: 'forward',
        onClick: () => communicationFormRef.current?.replyByMail(mailId, 'forward'),
        children: 'Переслать',
      });
    }

    if (isEditAccess(access.canForwardInternal, maxAccess)) {
      actions.push({
        key: 'forwardInternal',
        onClick: () => communicationFormRef.current?.replyByMail(mailId, 'forwardInternal'),
        children: 'Смежникам',
        title: 'Переслать смежникам',
      });
    }

    if (isEditAccess(access.canPrintAttachments, maxAccess)) {
      actions.push({
        key: 'printAttachments',
        onClick: () => {
          jsonApiCall({
            url: `/print/mail/${mailId}`,
          });
        },
        children: 'Напечатать все вложения',
      });
    }

    if (isEditAccess(access.canDetach, maxAccess)) {
      actions.push({
        key: 'detach',
        onClick: () =>
          allModals.detachTicket.open({
            mailId,
            issueId,
            onSubmit: (data) => {
              const value = Object(data.value);

              jsonApiCall({
                url: '/issue/detachmail',
                data: {
                  issueId,
                  mailId,
                  queueId: Object(value.queue).id,
                  accountId: Object(value.account).id,
                  categoryId: Object(value.category).id,
                },
              });
            },
            workplace: 'detachMail',
            eId: issueId,
            eType: ETypeString.Issue,
          }),
        icon: (cls) => <Icon2 className={cls} svg="split" />,
        title: 'Отделить',
      });
    }

    if (isReadAccess(access.access, maxAccess)) {
      actions.push({
        key: 'downloadMail',
        type: 'link',
        url: `${window.CRM_SPACE_API_HOST}/download/mail/${mailId}/${mailId}.msg`,
        icon: (cls) => <Icon2 className={cls} svg="download" />,
        title: 'Скачать письмо',
      });
    }

    if (isEditAccess(access.canCreateTicketST, maxAccess)) {
      actions.push({
        key: 'startrek',
        onClick: () =>
          allModals.createStartrek.open({
            mailId,
            issueId,
            onSubmit: (data) => {
              jsonApiCall({
                url: '/issue/startrek/create',
                data: {
                  id: issueId,
                  ...data,
                },
              });
            },
          }),
        icon: (cls) => <Icon2 className={cls} svg="startrek-color" />,
        title: 'Создать тикет Ст',
      });
    }

    if (isEditAccess(access.canCreateTicket, maxAccess)) {
      actions.push({
        key: 'request',
        onClick: () =>
          modal.open({
            data: {
              data: {
                mailId,
                issueId,
                byEObject: { eid: issueId, etype: ETypeString.Issue },
              },
              // TODO: https://st.yandex-team.ru/CRM-12496
              context: { module: AppModuleString.Ticket, action: 'createTicket2LineAction' },
            },
          }),
        icon: (cls) => <Icon2 className={cls} svg="bookmark" />,
        title: 'Создать тикет на 2 линию',
      });
    }
  }

  return actions;
};
