import React, { useRef, useState, useMemo } from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import Popup from '@crm/components/dist/lego2/Popup';
import Tooltip from '@crm/components/dist/lego2/Tooltip';
import { Text } from '@crm/components/dist/lego2/Text';
import { TimelineItemAlignment } from 'types/TimelineItemAlignment';
import { getCategories } from './Categorization.utils';
import { CategorizationProps } from './Categorization.types';
import css from './Categorization.module.css';

export const Categorization: React.FC<CategorizationProps> = (props) => {
  const {
    onClick,
    categorization,
    isRequireMarkup,
    alignment = TimelineItemAlignment.Right,
  } = props;

  const anchorRef = useRef<HTMLElement>(null);
  const [isPopupVisible, setIsPopupVisible] = useState(false);

  const showPopup = () => setIsPopupVisible(true);
  const hidePopup = () => setIsPopupVisible(false);

  const categories = useMemo(() => (categorization ? getCategories(categorization) : []), [
    categorization,
  ]);
  const hasCategories = categories.length > 0;
  const needMarkup = isRequireMarkup && !hasCategories;

  return (
    <div>
      <Button
        onClick={onClick}
        innerRef={anchorRef}
        view="clear"
        onMouseEnter={showPopup}
        onMouseLeave={hidePopup}
        icon={(cls) => (
          <>
            <Icon svg="tag" svgSize="ml" className={cls} />
            {hasCategories && (
              <Text
                typography="body-long-m"
                weight="medium"
                className={cx(cls, css.Categorization__count)}
              >
                {categories.length}
              </Text>
            )}
          </>
        )}
        addonAfter={needMarkup && <div className={css.Categorization__needMarkup} />}
      />

      {hasCategories && (
        <Popup
          view="default"
          className={css.Categorization__popup}
          visible={isPopupVisible}
          target="anchor"
          anchor={anchorRef}
          direction={alignment === TimelineItemAlignment.Right ? 'bottom-end' : 'bottom-start'}
        >
          {categories.map((category) => (
            <Text
              as="div"
              key={category.id}
              typography="body-short-m"
              className={css.Categorization__category}
            >
              {category.name}
            </Text>
          ))}
        </Popup>
      )}

      {needMarkup && (
        <Tooltip
          view="inverse"
          visible={isPopupVisible}
          hasTail
          size="s"
          anchor={anchorRef}
          direction={['top-center', 'bottom-center']}
        >
          Необходимо указать категорию
        </Tooltip>
      )}
    </div>
  );
};
