import {
  VirtualListDataProvider,
  VirtualListGetOptions,
  VirtualList,
  VirtualListMeta,
} from 'types/VirtualList';
import { get } from 'api/common';
import { getWikiFormatter } from 'utils/getWikiFormatter';
import { config } from 'services/Config';
import { VirtualListTimelineItem, VirtualListTimelineItemComment } from './TimelineV2.types';

export class IssueListApiService extends EventTarget
  implements VirtualListDataProvider<VirtualListTimelineItem> {
  constructor(private issueId: number) {
    super();

    if (!config.value.features.useYfm) {
      getWikiFormatter().then((wikiFormatter) => {
        this.wikiFormatter = wikiFormatter;
      });
    }
  }

  private wikiFormatter?: Ya.FormatterViewPort;

  get(options: VirtualListGetOptions): Promise<VirtualList<VirtualListTimelineItem>> {
    return get({
      url: `/issues/${this.issueId}/timeline`,
      data: options,
      showBackendErrorForUser: false,
    }).then(async (data: VirtualList<VirtualListTimelineItem>) => {
      if (!config.value.features.useYfm) {
        await Promise.all(
          data.data
            .filter((item) => item.etype === 'OComment')
            .map((comment) =>
              this.renderHtmlForComment((comment as unknown) as VirtualListTimelineItemComment),
            ),
        );
      }

      return data;
    });
  }

  private renderHtmlForComment(comment: VirtualListTimelineItemComment) {
    if (comment.data?.data?.text) {
      return this.wikiFormatter?.renderToString(comment.data.data.text).then((result) => {
        comment.data.data.html = result;
      });
    }
  }

  getMeta(): Promise<VirtualListMeta> {
    return get({ url: `/issues/${this.issueId}/timeline/meta`, showBackendErrorForUser: false });
  }
}
