import React, { useContext } from 'react';
import { history } from 'appHistory';
import { useContextState } from 'modules/categorization2';
import { ErrorBoundary } from 'components/ErrorBoundary';
import { Divider } from '../Divider';
import css from '../TimelineV2.module.css';
import { createAnyActions } from '../ActionsBuilder';
import { createAnySideActions } from '../SideActionsBuilder';
import { createAnyStatusActions } from '../StatusActionBuilder';
import { VirtualListTimelineItem } from '../TimelineV2.types';
import { issueTimelineItemContext } from '../issueTimelineItemContext';
import { ErrorItem, ErrorCode } from '../ErrorItem';
import { timelineItemToTimelineId } from '../utils/timelineItemToTimelineId';
import { useCategorizationHandlers } from '../../Timeline/Categorization/useCategorizationHandlers';
import { createDescription } from './createDescription';
import { ETypeToComponent, isSupportedType } from './ItemProxy';

export const Item = ({
  item,
  customTimelineId,
}: {
  item: VirtualListTimelineItem;
  customTimelineId?: string;
}) => {
  const {
    issueId,
    communicationFormRef,
    dataProvider,
    timelineId: contextTimelineId,
    moduleName,
    view,
    maxAccess,
  } = useContext(issueTimelineItemContext);

  const dataProviderItemData = dataProvider.getItemById(item.id);

  const categorizationStore = useContextState();

  const {
    onLoad: load,
    onTipLoad: loadTip,
    onAlertsLoad: loadAlerts,
    onTipComment: commentSubmit,
    onSearch: search,
    targetMeta,
    onChange: save,
  } = useCategorizationHandlers({
    issueId,
    eType: item.etype,
    eId: item.eid,
    initialValue: dataProviderItemData?.data?.categorization,
    targetDescription: createDescription(dataProviderItemData),
  });

  const renderContent = () => {
    const dataProviderItemData = dataProvider.getItemById(item.id);

    if (!dataProviderItemData) {
      return <ErrorItem error={{ message: 'No data in data provider', code: ErrorCode.NoData }} />;
    }

    if (!isSupportedType(dataProviderItemData.etype)) {
      return (
        <ErrorItem error={{ message: 'Unsupported type', code: ErrorCode.UnsupportedEType }} />
      );
    }

    if (dataProviderItemData.error) {
      return <ErrorItem error={dataProviderItemData.error} />;
    }

    if (!dataProviderItemData.data) {
      return <ErrorItem error={{ message: 'No data', code: ErrorCode.NoData }} />;
    }

    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const SupportedComponent = ETypeToComponent[dataProviderItemData.etype] as any;

    const data = dataProviderItemData.data.data;

    const timelineId = customTimelineId || timelineItemToTimelineId(dataProviderItemData);
    const timelineUrl = `/${moduleName}/${issueId}/${timelineId}${history.location.search}`;

    const handleCategorizationClick = () => {
      categorizationStore.open({
        previewComponent: () => (
          <div className={css.TimelineV2__itemPreview}>
            <SupportedComponent data={data} />
          </div>
        ),
        load,
        loadTip,
        loadAlerts,
        commentSubmit,
        search,
        save,
        targetMeta,
      });
    };

    const renderErrorItemFallback = (error) => (
      <ErrorItem error={{ message: error.message, code: ErrorCode.Error }} />
    );
    const showTags = moduleName === 'opportunities';

    return (
      <ErrorBoundary renderFallback={renderErrorItemFallback}>
        <SupportedComponent
          highlight={timelineId === contextTimelineId}
          itemId={item.id}
          showTags={showTags}
          tags={item?.data?.tags}
          linkToTimelineItem={timelineUrl}
          data={data}
          upsaleFactors={dataProviderItemData.data.upsaleFactors}
          mainActions={createAnyActions({
            data: dataProviderItemData,
            communicationFormRef,
            issueId,
            onCategorizationClick: handleCategorizationClick,
            maxAccess,
          })}
          sideActions={createAnySideActions({
            data: dataProviderItemData,
            communicationFormRef,
            issueId,
            onCategorizationClick: handleCategorizationClick,
            maxAccess,
          })}
          statusActions={createAnyStatusActions({
            data: dataProviderItemData,
            communicationFormRef,
            issueId,
            onCategorizationClick: handleCategorizationClick,
            maxAccess,
          })}
          alignment={item.alignment}
          backgroundColor={item.backgroundColor}
        />
      </ErrorBoundary>
    );
  };

  const prevItem = dataProvider.getItemById(dataProviderItemData!.prevId);
  const date = dataProviderItemData?.createdOn;
  const prevDate = prevItem ? prevItem.createdOn : undefined;

  return (
    <div className={css.TimelineV2__itemWrap}>
      <Divider
        className={css[`TimelineV2__itemDate_view_${view}`]}
        prevDate={prevDate}
        date={date}
      />
      {renderContent()}
    </div>
  );
};
