import React, { useContext } from 'react';
import { observer } from 'mobx-react-lite';
import { ETypeString } from 'types/entities';
import { TimelineMail } from 'components/TimelineMail';
import { TimelineComment } from 'components/TimelineComment';
import { TimelineMessage } from 'components/TimelineMessage';
import {
  VirtualListTimelineItem,
  VirtualListTimelineItemManagerActivity,
} from '../TimelineV2.types';
import { issueTimelineItemContext } from '../issueTimelineItemContext';
import { Item } from './Item';
import { ErrorCode, ErrorItem } from '../ErrorItem';
import { Call } from '../Call';

export const isManagerActivity = (
  dataProviderItemData: VirtualListTimelineItem,
): dataProviderItemData is VirtualListTimelineItemManagerActivity =>
  dataProviderItemData.etype === ETypeString.ManagerActivity;

export const ETypeToComponent = {
  [ETypeString.Mail]: TimelineMail,
  [ETypeString.OComment]: TimelineComment,
  [ETypeString.YcCall]: Call,
  [ETypeString.ChatMessage]: TimelineMessage,
  [ETypeString.InternalChatMessage]: TimelineMessage,
  [ETypeString.ManagerActivity]: ErrorItem,
} as const;

export const isSupportedType = (type: string) => Boolean(ETypeToComponent[type]);

export const ItemProxy = observer(({ item }: { item: VirtualListTimelineItem }) => {
  const { dataProvider } = useContext(issueTimelineItemContext);

  const dataProviderItemData = dataProvider.getItemById(item.id);

  if (!dataProviderItemData) {
    return <ErrorItem error={{ message: 'No data in data provider', code: ErrorCode.NoData }} />;
  }

  if (isManagerActivity(dataProviderItemData)) {
    if (dataProviderItemData.error || !dataProviderItemData.data) {
      return <Item item={item} />;
    }

    const innerData = {
      ...dataProviderItemData,
      ...dataProviderItemData.data.data,
    };

    innerData.data.props.category = 0;
    innerData.data.tags = item?.data?.tags;
    dataProvider.setItemById(innerData.id, innerData);
    return (
      innerData && (
        <Item
          item={innerData}
          customTimelineId={`${dataProviderItemData.etype}-${dataProviderItemData.eid}`}
        />
      )
    );
  }

  return <Item item={item} />;
});
