import React, { FC, useMemo, useRef, useEffect, useState } from 'react';
import { throttleRaf } from '@yandex-int/throttle';
import { VirtualList } from 'components/VirtualList';
import { FloatingDate } from 'components/FloatingDate';
import { FloatingDateInstance } from 'components/FloatingDate';
import { TimelineV2Props, VirtualListTimelineItem } from '../TimelineV2.types';
import { issueTimelineItemContext } from '../issueTimelineItemContext';
import { ItemProxy } from './ItemProxy';

const renderItemContent = (item: VirtualListTimelineItem) => <ItemProxy item={item} />;

const getNodeHeight = (node: HTMLElement) => (node ? node.offsetHeight : 0);

function getVisibleNodesRange(
  nodes: HTMLElement[],
  scrollerHeight: number,
  contentHeight: number,
  bottomOffset: number,
) {
  let begin = 0;
  let beginHeight = 0;
  let visibleHeight = 0;
  let hiddenHeight = 0;

  for (; begin < nodes.length; begin++) {
    const nodeHeight = getNodeHeight(nodes[begin]);
    beginHeight += nodeHeight;

    if (beginHeight > contentHeight - scrollerHeight - bottomOffset) {
      visibleHeight = beginHeight - (contentHeight - scrollerHeight - bottomOffset);
      hiddenHeight = nodeHeight - visibleHeight;

      break;
    }
  }

  return [begin, visibleHeight, hiddenHeight];
}

export const List: FC<TimelineV2Props> = ({
  listService,
  issueId,
  dataProvider,
  communicationFormRef,
  timelineId,
  moduleName,
  style,
  className,
  view,
  maxAccess,
  ...other
}) => {
  const floatingDateRef = useRef<FloatingDateInstance>(null);
  const [scrollerNode, setScrollerNode] = useState<HTMLElement | null>(null);
  const [containerNode, setContainerNode] = useState<HTMLElement | null>(null);

  const issueTimelineItemContextValue = useMemo(
    () => ({
      issueId,
      dataProvider,
      communicationFormRef,
      timelineId,
      moduleName,
      view,
      maxAccess,
    }),
    [issueId, dataProvider, communicationFormRef, timelineId, moduleName, view],
  );

  const handleScrollerRef = (ref: HTMLElement) => {
    setScrollerNode(ref);
    setContainerNode((ref?.children[0].children[1] as HTMLDivElement) || null);
  };

  useEffect(() => {
    const handleWheel = throttleRaf(() => {
      const nodes = containerNode!.children || [];
      if (!nodes.length) {
        return;
      }

      const scrollerHeight = scrollerNode!.clientHeight;
      const contentHeight =
        containerNode!.clientHeight - Number.parseFloat(containerNode!.style.paddingTop);
      const bottomOffset = Math.max(
        scrollerNode!.scrollHeight - scrollerNode!.scrollTop - scrollerNode!.clientHeight,
        Number.parseFloat(containerNode!.style.paddingBottom),
      );
      const [begin, firstVisibleHeight, firstHiddenHeight] = getVisibleNodesRange(
        (nodes as unknown) as HTMLElement[],
        scrollerHeight,
        contentHeight,
        bottomOffset,
      );
      const firstVisibleIndex = Number((nodes[begin] as HTMLElement)?.dataset?.itemIndex || -1);

      const firstVisibleItem = listService.getItemByAbsoluteIndex(firstVisibleIndex);
      const prevVisibleItem = listService.hasItemByAbsoluteIndex(firstVisibleIndex - 1)
        ? listService.getItemByAbsoluteIndex(firstVisibleIndex - 1)
        : undefined;
      const nextVisibleItem = listService.hasItemByAbsoluteIndex(firstVisibleIndex + 1)
        ? listService.getItemByAbsoluteIndex(firstVisibleIndex + 1)
        : undefined;

      const firstVisibleDate = firstVisibleItem.createdOn || '';
      const prevVisibleDate = prevVisibleItem ? prevVisibleItem.createdOn : undefined;
      const nextVisibleDate = nextVisibleItem ? nextVisibleItem.createdOn : undefined;

      floatingDateRef.current?.handleScroll?.({
        firstVisibleDate,
        prevVisibleDate,
        nextVisibleDate,
        firstVisibleHeight,
        firstHiddenHeight,
        bottomOffset,
      });
    });

    scrollerNode?.addEventListener('wheel', handleWheel);

    return () => {
      handleWheel.cancel();
      scrollerNode?.removeEventListener('wheel', handleWheel);
    };
  }, [scrollerNode, containerNode, listService]);

  return (
    <issueTimelineItemContext.Provider value={issueTimelineItemContextValue}>
      <FloatingDate ref={floatingDateRef} />
      <VirtualList<VirtualListTimelineItem>
        className={className}
        style={style}
        listService={listService}
        itemContent={renderItemContent}
        data-testid="timeline"
        {...other}
        scrollerRef={handleScrollerRef}
      />
    </issueTimelineItemContext.Provider>
  );
};
