import React, { useMemo, useRef, useContext, useEffect } from 'react';
import { runInAction } from 'mobx';
import { VirtualListService } from 'services/VirtualListService';
import cx from 'classnames';
import { XivaContext, XivaBackendEventType, IssueTimelineUpdateData } from 'modules/xiva';
import { AttachFilesDNDContainer } from 'components/AttachFiles';
import { TimelineCommunicationForm } from '../TimelineCommunicationForm';
import { List } from './List';
import { CommunicationForm } from './types/CommunicationForm';
import { TimelineV2IssueProps } from './TimelineV2.types';
import { IssueListApiService } from './IssueListApiService';
import { IssueListDataProviderService } from './IssueListDataProviderService';
import css from './TimelineV2.module.css';
import { timelineItemToTimelineId } from './utils/timelineItemToTimelineId';

export const TimelineV2 = ({
  issueId,
  className,
  timelineId,
  moduleName,
  maxAccess,
  height,
  afterContent,
  forceUseFeatureIssueTimelineV2,
  showCommentForm,
  showMailForm,
  showMessageForm,
  showInternalMessageForm,
  view = 'default',
}: TimelineV2IssueProps) => {
  const xiva = useContext(XivaContext);
  const communicationFormRef = useRef<CommunicationForm>(null);

  const dataProviderImpl = useMemo(() => {
    const listApiService = new IssueListApiService(issueId);

    return new IssueListDataProviderService(listApiService);
  }, [issueId]);

  const listService = useMemo(() => new VirtualListService(dataProviderImpl), [dataProviderImpl]);

  useEffect(() => {
    const update = (event: CustomEvent<IssueTimelineUpdateData>) => {
      if (event.detail.issueId !== issueId) {
        return;
      }

      event.detail.data.forEach((item) => {
        dataProviderImpl.setItemById(item.id, item);
      });

      runInAction(() => {
        listService.appendNonOrderItems(event.detail.data);

        if (event.detail.meta) {
          listService.updateMeta(event.detail.meta);
        }
      });
    };

    xiva.addEventListener(XivaBackendEventType.IssueTimelineUpdate, update);

    return () => {
      xiva.removeEventListener(XivaBackendEventType.IssueTimelineUpdate, update);
    };
  }, [dataProviderImpl, listService, issueId]);

  useEffect(() => {
    if (timelineId) {
      listService.init({
        from: 'center',
        exactId: timelineId,
        getInitialTopMostItemId: (items) => {
          const item = items.find((item) => timelineItemToTimelineId(item) === timelineId);
          return item ? item.id : null;
        },
      });
    } else {
      listService.init({ from: 'end' });
    }

    return () => {
      listService.destroy();
    };
  }, [listService]);

  const listStyle = height ? { height: `${height}px`, flex: 'auto' } : undefined;

  return (
    <AttachFilesDNDContainer
      className={cx(className, css.TimelineV2, css[`TimelineV2_view_${view}`])}
    >
      <List
        className={css.TimelineV2__list}
        issueId={issueId}
        timelineId={timelineId}
        moduleName={moduleName}
        listService={listService}
        dataProvider={dataProviderImpl}
        communicationFormRef={communicationFormRef}
        style={listStyle}
        view={view}
        maxAccess={maxAccess}
      />
      {afterContent}
      <TimelineCommunicationForm
        className={css.TimelineV2__form}
        issueId={issueId}
        ref={communicationFormRef}
        maxAccess={maxAccess}
        forceUseFeatureIssueTimelineV2={forceUseFeatureIssueTimelineV2}
        showCommentForm={showCommentForm}
        showMailForm={showMailForm}
        showMessageForm={showMessageForm}
        showInternalMessageForm={showInternalMessageForm}
      />
    </AttachFilesDNDContainer>
  );
};
