import React, { CSSProperties } from 'react';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import InfiniteList from 'components/InfiniteList';
import { Node } from '../types';

const PADDING = { marginLeft: 20 };
const MORE = { display: 'flex', justifyContent: 'center' };

export interface ComponentProps {
  id: number;
  maxAccess?: number;
  style?: CSSProperties;
  isOpen?: boolean;
  more: React.ReactNode;
  onToggle?: () => void;
  moreToggle?: () => void;
  moreCounter: number;
  moreState: 'open' | 'close';
}

export type TreeComponentType = React.ComponentType<ComponentProps>;

export interface TreeNodeProps {
  id: number;
  isRootNode?: boolean;
  path: string;
  component?: TreeComponentType;
  // eslint-disable-next-line
  wrap?: React.ComponentType<any> | string;
  maxAccess?: number;
  // eslint-disable-next-line
  getLoadParamExtension?: () => any;
  className?: string;
  hasInfiniteScroll?: boolean;
}

export type TreeNodeType = React.ComponentType<TreeNodeProps>;

export interface WrapProps {
  header: React.ReactNode;
  isEof: boolean;
  onLoad: () => void;
  isLoading: boolean;
  isLoadingFirstPage?: boolean;
  offset: number;
  className?: string;
}

export interface NodeProps {
  node: Node;
  treeNode: TreeNodeType;
  wrap: React.ComponentType<Partial<WrapProps>> | string;
  isRootNode?: boolean;
  path: string;
  maxAccess?: number;
  toggle: () => void;
  more: () => void;
  hasInfiniteScroll: boolean;
  isFetch: boolean;
  offset: number;
  className?: string;
  header?: React.ReactNode;
  component?: TreeComponentType;
  // eslint-disable-next-line
  getLoadParamExtension?: () => any;
  isLoadingFirstPage?: boolean;
}

export type TreeNodeHOC = (WrappedComponent: React.FC<NodeProps>) => TreeNodeType;

export interface CreateTreeNodeConfig {
  hoc: TreeNodeHOC;
  component?: React.ComponentType<ComponentProps>;
  wrap?: React.ComponentType<WrapProps> | string;
}

const createTreeNode = (config: CreateTreeNodeConfig) => {
  const { hoc, ...defaultProps } = config;

  const TreeNodeDumb: React.FC<NodeProps> = (props) => {
    const {
      node,
      more,
      toggle,
      component: Component,
      treeNode: TreeNode,
      isRootNode,
      hasInfiniteScroll,
      wrap,
      className,
      header,
      isFetch,
      isLoadingFirstPage,
      offset,
      path,
      maxAccess,
    } = props;

    if (!node) {
      return null;
    }

    if (!Component) {
      return null;
    }

    const isInfiniteScroll = isRootNode && hasInfiniteScroll;

    const { id, eof, items = [], counter } = node;
    const IS_SUB_ITEMS = Boolean(items.length);

    const Wrap = isInfiniteScroll ? InfiniteList : wrap;
    // eslint-disable-next-line
    const wrapProps: WrapProps = ({} as any) as WrapProps;
    if (className) {
      wrapProps.className = className;
    }

    if (isInfiniteScroll) {
      wrapProps.isEof = eof;
      wrapProps.onLoad = more;
      wrapProps.header = header;
      wrapProps.isLoading = isFetch;
      wrapProps.isLoadingFirstPage = isLoadingFirstPage;
      wrapProps.offset = offset;
    }

    const moreToggle = IS_SUB_ITEMS ? toggle : more;

    return (
      <Wrap {...wrapProps}>
        {!isRootNode && (
          <Component
            id={id}
            maxAccess={maxAccess}
            more={
              Boolean(counter) && (
                <Button
                  size="xs"
                  view="pseudo"
                  onClick={moreToggle}
                  disabled={isFetch}
                  progress={isFetch}
                  iconRight={(iconCls) => (
                    <Icon
                      className={iconCls}
                      type="arrow"
                      direction={IS_SUB_ITEMS ? 'top' : 'bottom'}
                    />
                  )}
                >
                  {counter}
                </Button>
              )
            }
            moreToggle={counter ? moreToggle : undefined}
            moreCounter={counter}
            moreState={IS_SUB_ITEMS ? 'open' : 'close'}
          />
        )}
        <div style={isRootNode ? undefined : PADDING}>
          <div>
            {items.map((item) => (
              <TreeNode
                key={item}
                id={item}
                isRootNode={false}
                path={path}
                component={Component}
                wrap={wrap}
                maxAccess={maxAccess}
              />
            ))}
          </div>
          <div style={MORE}>
            {!isInfiniteScroll && IS_SUB_ITEMS && !eof && (
              <Button size="xs" onClick={more} disabled={isFetch} progress={isFetch}>
                more
              </Button>
            )}
          </div>
        </div>
      </Wrap>
    );
  };

  const ConnectedTreeNodeDumb = hoc(TreeNodeDumb);

  TreeNodeDumb.defaultProps = {
    treeNode: ConnectedTreeNodeDumb,
    wrap: 'div',
    isRootNode: true,
    ...defaultProps,
  };

  return ConnectedTreeNodeDumb;
};

export default createTreeNode;
