import React, { ComponentType, forwardRef, useContext } from 'react';
import createRedux from './createRedux';
import { Redux, InjectedReduxContextProps } from './types';
import { ModuleName } from '../types';

export const ReduxContext = React.createContext<Redux>((null as unknown) as Redux);

export const ReduxContextProvider = (props) =>
  React.createElement(
    ReduxContext.Provider,
    {
      value: createRedux(props.module),
    },
    props.children,
  );

interface WithReduxProviderConfig {
  module?: ModuleName;
}

export const withReduxProvider = <P extends {}>(config?: WithReduxProviderConfig) => (
  WrappedComponent: ComponentType<P>,
) => {
  const WithReduxProvider = (props: WithReduxProviderConfig & P) => {
    const { module } = props;

    return (
      <ReduxContextProvider module={module}>
        <WrappedComponent {...(props as P)} />
      </ReduxContextProvider>
    );
  };

  WithReduxProvider.defaultProps = {
    ...config,
    module: ModuleName.Issue,
  };

  return WithReduxProvider;
};

export function withReduxContext<P extends {}>(
  WrappedComponent: ComponentType<P & InjectedReduxContextProps>,
) {
  class WithReduxContext extends React.Component<P> {
    public static contextType = ReduxContext;

    public context!: Redux;

    public render() {
      return React.createElement<P & InjectedReduxContextProps>(WrappedComponent, {
        redux: this.context,
        ...(this.props as P),
      });
    }
  }

  return WithReduxContext;
}

export function withReduxContextWithForwardRef<P extends {}>(
  WrappedComponent: ComponentType<P & InjectedReduxContextProps>,
) {
  return forwardRef<typeof WrappedComponent, P>((props, ref) => {
    const redux = useContext(ReduxContext);

    return React.createElement<P & InjectedReduxContextProps>(WrappedComponent, {
      ref,
      redux,
      ...(props as P),
    });
  });
}
