import combineReducers from 'utils/reducer/combineReducers';
import createStorageSlice from './slices/createStorageSlice';
import createIssueSlice from './slices/createIssueSlice';
import createTimelineSettingsSlice from './slices/createTimelineSettingsSlice';
import createMainSlice from './slices/createMainSlice';
import createNodesSlice from './slices/createNodesSlice';
import createTimelineSlice from './slices/createTimelineSlice';
import createModuleSlice from './slices/createModuleSlice';
import createFilterSlice from './slices/createFilterSlice';
import createSliceMapReducer from './createSliceMapReducer';
import createSelectors from './createSelectors';
import { ModuleName } from '../types';

const createRedux = (name: ModuleName) => {
  const selectors = createSelectors(name);

  const options = {
    name,
    selectors,
    slices: {
      storageSlice: {},
      nodesSlice: {},
    },
  };

  const storageSlice = createStorageSlice(options);

  options.slices.storageSlice = storageSlice;

  const issueSlice = createIssueSlice(options);
  const timelineSettingsSlice = createTimelineSettingsSlice(options);
  const mainSlice = createMainSlice(options);
  const nodesSlice = createNodesSlice(options);

  options.slices.nodesSlice = nodesSlice;

  const timelineSlice = createTimelineSlice(options);
  const moduleSlice = createModuleSlice(options);
  const filterSlice = createFilterSlice(options);

  const reducer = (state, action) => {
    if (action.type === 'LOCAL_STORAGE' || action.type === moduleSlice.actions.destroy.toString()) {
      return combineReducers({
        filter: filterSlice.reducer,
        timeline: timelineSettingsSlice.reducer,
      })(state, { type: 'LOCAL_STORAGE' });
    }

    return combineReducers({
      storage: storageSlice.reducer,
      filter: filterSlice.reducer,
      timeline: timelineSettingsSlice.reducer,
      issues: createSliceMapReducer(issueSlice, 'id'),
      module: moduleSlice.reducer,
      meta: combineReducers({
        nodes: createSliceMapReducer(nodesSlice, 'id'),
        timelines: createSliceMapReducer(timelineSlice, 'id'),
        main: mainSlice.reducer,
      }),
    })(state, action);
  };

  window.reduxStore.injectReducerWithLocalStorage(name, reducer);

  const redux = {
    name,
    slices: {
      storageSlice,
      issueSlice,
      timelineSettingsSlice,
      mainSlice,
      nodesSlice,
      timelineSlice,
      moduleSlice,
      filterSlice,
    },
    selectors,
    reducer,
  };

  return redux;
};

export type Redux = ReturnType<typeof createRedux>;
export type State = ReturnType<ReturnType<typeof createRedux>['reducer']>;

const cache: { [key: string]: ReturnType<typeof createRedux> } = {};

const createReduxWithCache = (name: ModuleName) => {
  if (cache[name]) {
    return cache[name];
  }

  const redux = createRedux(name);

  cache[name] = redux;

  return redux;
};

export default createReduxWithCache;
