import get from 'lodash/get';
import memoize from 'memoize-one';
import { isTeamleadOrGrouplead } from 'modules/initial/selectors';
import storagePathCreator, { linkIssueKeys, similarIssueKeys } from '../utils/storagePathCreator';
import { ModuleBackendMapping } from '../types';

// TODO: https://st.yandex-team.ru/CRM-10561
export const isTestMode = (state) => state.root.info.testMode;

const selectorFactory = (path) => ({
  getAllNodes(state) {
    return get(state, [path, 'storage', 'nodes']);
  },
  getAll(state, slug) {
    return get(state, [path, 'storage', 'nodes', ...slug.split('.'), 'map']);
  },
  getItem(state, slug, id) {
    return get(state, [path, 'storage', 'nodes', ...slug.split('.'), 'map', id]);
  },
  getField(state, slug, id, name) {
    return get(state, [path, 'storage', 'nodes', ...slug.split('.'), 'map', id, name]);
  },
});

const createDataSelector = (path, slug) => ({
  getAll(state) {
    return get(state, [path, 'storage', slug]);
  },
  getItem(state, id) {
    return get(state, [path, 'storage', slug, id]);
  },
  getScheme(state, id) {
    return get(state, [path, 'storage', slug, id, 'scheme']);
  },
  getField(state, id, name) {
    return get(state, [path, 'storage', slug, id, 'data', name]);
  },
  getFieldAccess(state, id, name) {
    return get(state, [path, 'storage', slug, id, 'props', name, 'access']);
  },
  getFieldCounter(state, id, name): number | undefined {
    return get(state, [path, 'storage', slug, id, 'props', name, 'counter']);
  },
});

const createFilterSelectors = (path) => {
  const getFastFilter = (state) => {
    let filter = get(state, `${path}.filter.filters.fast`);

    if (filter && filter.panel === 'MyGroup' && !isTeamleadOrGrouplead(state)) {
      filter = { ...filter, panel: 'My' };
    }

    return filter;
  };

  const getSearchFilterSort = (state) => {
    return get(state, `${path}.filter.sort`);
  };

  const getSearchFilterDataUI = (state) => {
    return get(state, `${path}.filter.filters.search.valuesUI`);
  };

  const getSearchFilterDataBackend = (state) => {
    return get(state, `${path}.filter.filters.search.values`);
  };

  const getCurrentFilterType = (state) => {
    return get(state, `${path}.filter.currentFilter`);
  };

  const currentFilterDataMappper = {
    search: getSearchFilterDataBackend,
    fast: getFastFilter,
  };

  const getCurrentFilterData = (state) => {
    const func = currentFilterDataMappper[getCurrentFilterType(state)];
    const sort = getSearchFilterSort(state);
    const filterValue = func ? func(state) : undefined;

    return { ...filterValue, sort };
  };

  const formatSearcFilterUrl = memoize((filter) => ({ search: filter }));

  const getSearchFilterUrl = (state) => {
    return formatSearcFilterUrl(get(state, `${path}.filter.filters.search`));
  };

  const currentFilterDataUrlMappper = {
    search: getSearchFilterUrl,
    fast: getFastFilter,
  };

  const getCurrentFilterDataForUrl = (state) => {
    const func = currentFilterDataUrlMappper[getCurrentFilterType(state)];
    return func ? func(state) : undefined;
  };

  return {
    getFastFilter,
    getSearchFilterSort,
    getSearchFilterDataUI,
    getSearchFilterDataBackend,
    getCurrentFilterType,
    getCurrentFilterData,
    getCurrentFilterDataForUrl,
  };
};

const createSelectors = (path) => {
  const selectors = {
    storage: {
      managerActivities: createDataSelector(path, 'managerActivities'),
      calls: createDataSelector(path, 'calls'),
      comments: createDataSelector(path, 'comments'),
      messages: createDataSelector(path, 'messages'),
      mails: createDataSelector(path, 'mails'),
      meetings: createDataSelector(path, 'meetings'),
      issues: createDataSelector(path, 'issues'),
      issueFilters: createDataSelector(path, 'issueFilters'),
      startrekIssues: createDataSelector(path, 'startrekIssues'),
      nodes: selectorFactory(path),
      getTakeNextAvailableCount: (state) =>
        get(state, `${path}.storage.counters.issue.takeNextAvailableCount`),
    },
    meta: {
      nodes: (state, id) => get(state, [path, 'meta', 'nodes', id]),
      filterNodes: (state, id) => get(state, [path, 'meta', 'filterNodes', id]),
      timelines: (state, id) => get(state, [path, 'meta', 'timelines', id]),
    },
    getState(state) {
      return get(state, path);
    },
    getFilter(state) {
      let filter = get(state, `${path}.filter.filters.fast`);

      if (filter && filter.panel === 'MyGroup' && !isTeamleadOrGrouplead(state)) {
        filter = { ...filter, panel: 'My' };
      }

      return filter;
    },
    timeline: {
      sort(state) {
        return get(state, [path, 'timeline', 'sort']);
      },
    },
    getMainId(state) {
      return get(state, [path, 'meta', 'main']);
    },
    isIssueLoad(state, id) {
      return get(state, [path, 'issues', id, 'isLoad']);
    },
    isIssueFetch(state, id) {
      return get(state, [path, 'issues', id, 'isFetch']);
    },
  };

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  const getLinkGroupCount = (state: any, linkIssuePath: string): number => {
    const node = selectors.storage.nodes.getItem(state, linkIssuePath, 0) as
      | { items: unknown[] }
      | undefined;

    return node?.items?.length || 0;
  };

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  const getLinksCount = (state: any, issueId: number) => {
    return linkIssueKeys
      .map((key) => getLinkGroupCount(state, storagePathCreator[key](issueId)))
      .reduce((ac, v) => ac + v, 0);
  };

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  const getSimilarCount = (state: any, issueId: number) => {
    return similarIssueKeys
      .map((key) => getLinkGroupCount(state, storagePathCreator[key](issueId)))
      .reduce((ac, v) => ac + v, 0);
  };

  return {
    ...selectors,
    getLinksCount,
    getSimilarCount,
  };
};

export default (name: string) => {
  const selectors = createSelectors(name);
  const filterSelectors = createFilterSelectors(name);

  const moduleSettings = (state) => get(state, [name, 'module']);

  const context = (state) => {
    const mainId = selectors.getMainId(state);

    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const ctx: any = {};

    if (mainId) {
      ctx.main = {
        issue: {
          id: mainId,
        },
      };
    }

    ctx.nodes = selectors.storage.nodes.getAllNodes(state);
    // todo: что делаем с фильтрами?
    ctx.filters = {
      issueList: filterSelectors.getCurrentFilterData(state),
    };

    ctx.module = ModuleBackendMapping[name];

    return ctx;
  };

  return {
    ...selectors,
    ...filterSelectors,
    context,
    moduleSettings,
  };
};
