/* eslint-disable no-param-reassign,@typescript-eslint/no-explicit-any */
import { createSlice } from '@reduxjs/toolkit';
import Adapter from 'utils/Adapter';
import downloadFile from 'utils/downloadFile';
import { history } from 'appHistory';
import { get } from 'api/common';
import browserWindow from 'utils/browserWindow';
import {
  getIssue,
  deleteIssue,
  mutationIssue,
  linkIssue,
  createStartrek,
  linkStartrek,
  unlinkStartrek,
  createIssue,
  detachMail,
  CreateIssueDataSome,
} from '../api';
import { StorageResponse, ModuleName } from '../../types';
import createAsyncAction from '../createAsyncAction';

// eslint-disable-next-line @typescript-eslint/no-explicit-any
type AppThunk = any;

const prepare = (id: number) => ({ payload: id, meta: { id } });

const withMeta = (reducer) => ({
  reducer,
  prepare,
});

type MutationAdapter = (id: number, data: any) => { api: any; sendingData: any };

const mutationAdapter = new Adapter<MutationAdapter>((id, data) => ({
  api: mutationIssue,
  sendingData: { id, ...data },
}));

mutationAdapter.add('linkIssue', (id, data) => ({
  api: linkIssue,
  sendingData: data.value,
}));

mutationAdapter.add('createStartrek', (id, data) => ({
  api: createStartrek,
  sendingData: { id, ...data },
}));

mutationAdapter.add('linkStartrek', (id, data) => ({
  api: linkStartrek,
  sendingData: { id, ...data },
}));

mutationAdapter.add('unlinkStartrek', (id, data) => ({
  api: unlinkStartrek,
  sendingData: { id, ...data },
}));

mutationAdapter.add('detach', (id, data = {}) => {
  const value = Object(data.value);

  return {
    api: detachMail,
    sendingData: {
      issueId: id,
      mailId: data.mailId,
      queueId: Object(value.queue).id,
      accountId: Object(value.account).id,
      categoryId: Object(value.category).id,
    },
  };
});

export default (options) => {
  const { name, slices, selectors } = options;

  const issueSlice = createSlice({
    name: `${name}/issue`,
    initialState: {
      isFetch: false,
      isLoad: false,
    },
    reducers: {
      getIssueRequest: withMeta((state) => {
        state.isFetch = true;
      }),
      getIssueSuccess: withMeta((state) => {
        state.isLoad = true;
        state.isFetch = false;
      }),
      getIssueFailed: withMeta((state) => {
        state.isFetch = false;
      }),
      deleteIssueSuccess: withMeta(() => {
        return undefined;
      }),
    },
  });

  const {
    getIssueRequest,
    getIssueSuccess,
    getIssueFailed,
    deleteIssueSuccess,
  } = issueSlice.actions;

  const fetchIssue = (id: number): AppThunk => async (dispatch, getState) => {
    try {
      dispatch(getIssueRequest(id));
      const data = await getIssue({ data: { id }, context: selectors.context(getState()) });
      if (data.storage) {
        dispatch(slices.storageSlice.actions.update(data.storage));
        dispatch(slices.storageSlice.asyncActions.updateModuleCounter(data.storage));
      }
      dispatch(getIssueSuccess(id));
    } catch (err) {
      dispatch(getIssueFailed(id));
    }
  };

  const fetchDeleteIssue = (id: number): AppThunk => async (dispatch, getState) => {
    try {
      const data = await deleteIssue({ data: { id }, context: selectors.context(getState()) });
      if (data.storage) {
        dispatch(slices.storageSlice.actions.update(data.storage));
        dispatch(slices.storageSlice.asyncActions.updateModuleCounter(data.storage));
      }
      dispatch(deleteIssueSuccess(id));
    } catch (err) {
      dispatch(getIssueFailed(id));
    }
  };

  const fetchMutationIssue = (id: number, data: any = {}): AppThunk => async (
    dispatch,
    getState,
  ) => {
    // eslint-disable-next-line no-shadow
    const { name: fieldName, options } = data;
    try {
      let response: StorageResponse;

      if (options && options.url) {
        // download files
        if (options.url.indexOf('download') !== -1) {
          downloadFile(options.url);

          return;
        }

        // handle action like "print files"
        response = await get({ url: options.url });
      } else {
        const { api, sendingData } = mutationAdapter.getOrDefault(fieldName)(id, data);

        response = await api({
          data: sendingData,
          context: selectors.context(getState()),
        });
      }

      if (response.storage) {
        dispatch(slices.storageSlice.actions.update(response.storage));
        dispatch(slices.storageSlice.asyncActions.updateModuleCounter(response.storage));
      }
    } catch (err) {
      throw err;
    }
  };

  const goToIssue = ({
    issueId,
    blank,
    moduleName = name,
  }: {
    issueId: string | number;
    blank: boolean;
    moduleName?: ModuleName;
  }) => {
    const search = history.location.search || '';

    const nextPathLocation = `/${moduleName}/${issueId}`;
    const nextLocation = `${nextPathLocation}${search}`;
    if (blank) {
      browserWindow.open(nextLocation);
    } else if (nextPathLocation !== history.location.pathname) {
      history.push(nextLocation);
    }
  };

  return {
    ...issueSlice,
    goToIssue,
    asyncActions: {
      fetchIssue,
      fetchDeleteIssue,
      fetchMutationIssue,
      createIssue: createAsyncAction<CreateIssueDataSome>(createIssue, options),
    },
  };
};
