/* eslint-disable no-param-reassign */
import { createSlice } from '@reduxjs/toolkit';
import { getIssues, getFilters, getStartrekIssues } from 'modules/issues/redux/api';
import storagePathCreator from '../../utils/storagePathCreator';

type AppThunk = any;

const prepare = (id: number, offset?: number) => ({ payload: offset, meta: { id } });

const withMeta = reducer => ({
  reducer,
  prepare,
});

export default options => {
  const { name, slices, selectors } = options;

  const nodesSlice = createSlice({
    name: `${name}/issueLists`,
    initialState: {
      isFetch: false,
      offset: 0,
    },
    reducers: {
      getIssueListRequest: withMeta((state, { payload }) => {
        state.isFetch = true;
        state.isLoadingFirstPage = !payload;
        state.offset = payload;
      }),
      getIssueListSuccess: withMeta(state => {
        state.isFetch = false;
        state.isLoadingFirstPage = false;
      }),
      getIssueListFailed: withMeta(state => {
        state.isFetch = false;
        state.isLoadingFirstPage = false;
      }),
    },
  });

  const promiseMap: any = {};

  const { getIssueListRequest, getIssueListSuccess, getIssueListFailed } = nodesSlice.actions;

  const createGet = (request, $path?: string | ((data: any) => string)) => (
    data: { [key: string]: any } = {},
    signal?: AbortSignal,
  ): AppThunk => async (dispatch, getState) => {
    let path = '';
    if (typeof $path === 'function') {
      path = $path(data);
    } else {
      path = $path || storagePathCreator.data2Path(data) || '';
    }
    // пока такой костыль для проверки отправки на бек фильтров только для issueList
    const isListIssue = storagePathCreator.isListIssuePath(path);
    const nodeId = data.nodeId || data.parentId || 0;

    const promiseMapSlice = promiseMap[path];
    if (promiseMapSlice) {
      if (!data.offset) {
        promiseMapSlice.forEach((value, key, map) => {
          value.cancel();
          map.delete(key);
        });
      } else if (promiseMapSlice.get(nodeId)) {
        promiseMapSlice.get(nodeId).cancel();
      }
    }

    try {
      const promise = request({
        data: {
          ...data,
          filter: isListIssue ? selectors.getCurrentFilterData(getState()) : undefined,
        },
        context: selectors.context(getState()),
      });

      if (!promiseMap[path]) {
        promiseMap[path] = new Map();
      }

      promiseMap[path].set(nodeId, promise);
      promise.finally(() => {
        if (promiseMap[path]) {
          promiseMap[path].delete(nodeId);
        }
      });

      dispatch(getIssueListRequest(nodeId, data.offset));
      const response = await promise;

      if (signal && signal.aborted) {
        return;
      }

      if (response.storage) {
        dispatch(slices.storageSlice.actions.update(response.storage));
        dispatch(slices.storageSlice.asyncActions.updateModuleCounter(response.storage));
      }
      dispatch(getIssueListSuccess(nodeId));
    } catch (err) {
      dispatch(getIssueListFailed(nodeId));
    }
  };

  const asyncActions = {
    get: createGet(getIssues),
    getFilters: createGet(getFilters, 'issueFilters'),
    getStartrekIssues: createGet(getStartrekIssues, data =>
      storagePathCreator.startrekIssues(data.issueId),
    ),
  };

  return {
    ...nodesSlice,
    asyncActions,
  };
};
