/* eslint-disable no-shadow */

import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { actions as moduleCountersActions, modules } from 'reducers/moduleCountersSlice';
import get from 'lodash/get';
import set from 'lodash/set';
import { IssueTimelineUpdateData } from 'modules/xiva/types/IssueTimelineUpdateData';
import { EObjectFormatter } from 'utils/EObjectFormatter';
import { ETypeString } from 'types/entities';
import { transformTimelineV1ToTimelineV2 } from '../../utils/transformTimelineV1ToTimelineV2';
import storageMerge from '../storageMerge';
import { StorageData } from '../../types';

// eslint-disable-next-line
type AppThunk = any;

const mapEtypeToStorageKey = {
  [ETypeString.YcCall]: 'calls',
  [ETypeString.ManagerActivity]: 'managerActivities',
};

const storageEObjectFormatter = new EObjectFormatter();

// eslint-disable-next-line
export default (options: any) => {
  const updateModuleCounter = (state: StorageData): AppThunk => (dispatch) => {
    if (!state.counters || !state.counters.issues) {
      return;
    }

    dispatch(
      moduleCountersActions.set({
        [modules.issue]: state.counters.issues.unread,
      }),
    );
  };

  const slice = createSlice({
    name: `${options.name}/storage`,
    initialState: Object() as StorageData,
    reducers: {
      update: (state, { payload }: PayloadAction<StorageData>) => {
        let newStorage = state;

        if (payload) {
          newStorage = storageMerge(state, payload);
        }

        return newStorage;
      },
      updateFromTimelineV2Data(state, { payload }: PayloadAction<IssueTimelineUpdateData>) {
        let map = get(state, ['nodes', 'issues', payload.issueId, 'timeline', 'map']);

        if (!map) {
          return state;
        }

        let map0 = map[0];
        if (!map0) {
          return state;
        }

        map0 = { ...map0, items: [].concat(map0.items || []) };
        map = { ...map, '0': map0 };

        const patch = { nodes: { issues: { [payload.issueId]: { timeline: { map } } } } };
        Object.values(mapEtypeToStorageKey).forEach((key) => {
          patch[key] = {};
        });

        let hasUpdates = false;

        payload.data.forEach((item) => {
          const etypeEid = storageEObjectFormatter.format(item);

          if (map[etypeEid] || !mapEtypeToStorageKey[item.etype]) {
            return;
          }

          hasUpdates = true;
          patch[mapEtypeToStorageKey[item.etype]][item.eid] = transformTimelineV1ToTimelineV2(item);
          map[etypeEid] = { counter: 0, eof: true, items: [], id: item.eid };
          map0.items.push(etypeEid);
          map0.counter += 1;
        });

        if (!hasUpdates) {
          return state;
        }

        return storageMerge(state, patch);
      },
      toggle: {
        reducer: (state, { meta: { name, path } }) => {
          const nodePath = `nodes.${path}.map.${name}`;
          let node = get(state, nodePath);
          if (node) {
            node = { ...node };

            delete node.items;
            delete node.eof;

            const patchStorage = {};
            set(patchStorage, nodePath, node);

            return storageMerge(state, patchStorage);
          }

          return state;
        },
        prepare(path, id) {
          return {
            payload: {},
            meta: {
              name: id,
              path,
            },
            error: undefined,
          };
        },
      },
    },
  });

  return {
    ...slice,
    asyncActions: {
      updateModuleCounter,
    },
  };
};
