import setWith from 'lodash/setWith';
import clone from 'lodash/clone';
import { stateModel } from 'services/DBProvider/Models/Model.config';
import { logger } from 'services/Logger';
import migration from './migration';

const getStorageKeyForRemove = (userId: number): string => `space.user_id_${userId}`;
const getStorageOldKey = (userId: number): string => `space.user_id_${userId}_v2`;
const getStorageKey = (userId: number): string => `space.user_id_${userId}_v3`;

let initState = {};

// eslint-disable-next-line @typescript-eslint/no-explicit-any
type StateSaverFunc = (state: any) => any;

interface StateSaver {
  list: StateSaverFunc[];
  addHandler: (func: StateSaverFunc) => void;
  removeHandler: (func: StateSaverFunc) => void;
}

export const stateSaver: StateSaver = {
  list: [],

  addHandler(func) {
    this.list.push(func);
  },
  removeHandler(func) {
    this.list = this.list.filter((item) => item !== func);
  },
};

const saveStateToLocalStorage = (state) => {
  try {
    const serializedState = JSON.stringify(state);
    localStorage.setItem(getStorageKey(window.USER_ID as number), serializedState);
  } catch (error) {
    logger.reportError(error);
  }
};

export const loadState = () => {
  try {
    const userId = window.USER_ID as number;

    window.localStorage.removeItem(getStorageKeyForRemove(userId));

    const serializedState = localStorage.getItem(getStorageKey(userId));
    const serializedOldState = localStorage.getItem(getStorageOldKey(userId));

    if (serializedState) {
      initState = JSON.parse(serializedState);

      return JSON.parse(serializedState);
    }
    if (serializedOldState) {
      initState = JSON.parse(serializedOldState);

      migration(initState);
      saveStateToLocalStorage(initState);

      return initState;
    }
  } catch (error) {
    logger.reportError(error);
  }

  return undefined;
};

export const getNewState = (state) => {
  let newState = initState || {};
  stateSaver.list.forEach((func) => {
    const result = func(state);
    if (result && result.path) {
      newState = setWith(clone(newState), result.path, result.state, clone);
    } else {
      newState = { ...newState, ...result };
    }
  });

  newState = {
    ...newState,
    ...window.reduxStore.localStorageReducer(state, { type: 'LOCAL_STORAGE' }),
  };

  return newState;
};

const filterObject = (obj: unknown) => {
  if (typeof obj !== 'object' || Array.isArray(obj) || obj === null) {
    return obj;
  }

  return Object.keys(obj).reduce((acc, key) => {
    if (obj[key] !== undefined) {
      acc[key] = filterObject(obj[key]);
    }
    return acc;
  }, {});
};

export const saveStateToIndexedDb = async (state) => {
  try {
    const model = await stateModel;
    model?.update(filterObject(getNewState(state)));
  } catch (error) {
    logger.reportError(error);
  }
};
