import { baseCallApi, jsonApiCall } from 'api/common';
import { history } from 'appHistory';
import { getFormValues, initialize } from 'redux-form';
import pick from 'lodash/pick';
import { reload } from 'containers/InfiniteList/actions';
import { push as notificationsPush } from 'modules/notificationsUI/actions';
import * as aTypes from '../actionTypes/massmail';
import { ROUTER_PATH } from '../constants';
import { getFirstContantId, getPreviewArgs } from '../selectors';

export const loadMassmail = (massmailId) => (dispatch) => {
  const url = massmailId ? '/view/massmail' : '/view/massmail/new';
  const api = massmailId ? baseCallApi : jsonApiCall;

  const promise = api({
    url,
    data: { massmailId },
  });

  if (!massmailId) {
    promise.then((data) => history.push(`${ROUTER_PATH}/${data.mail.id}`));
  }

  return dispatch({
    type: 'PROMISE',
    actions: [aTypes.MASSMAIL_REQUEST, aTypes.MASSMAIL_RECEIVE, aTypes.MASSMAIL_FAIL],
    promise,
  });
};

export const sendMassmail = (massmailId) => (dispatch) => {
  jsonApiCall({
    url: '/actions/massmail/send',
    data: { massmailId },
  }).then(() => {
    dispatch(notificationsPush({ theme: 'success', text: 'Рассылка отправлена' }));
    dispatch(loadMassmail(massmailId));
    dispatch(reload('MASS_MAIL_LIST'));
  });
};

export const updateMassmail = (massmailId) => ({
  type: 'PROMISE',
  actions: ['', aTypes.MASSMAIL_UPDATE_RECEIVE, ''],
  promise: baseCallApi({
    url: '/view/massmail',
    data: { massmailId },
  }).then((data) => pick(data, ['macros', 'recipientViewItem'])),
});

const previewStoreArgs = (payload) => ({ type: aTypes.PREVIEW_STORE_ARGS, payload });
const previewStoreRecipientId = (recipientId) => ({
  type: aTypes.PREVIEW_STORE_RECIPIENT_ID,
  payload: recipientId,
});

export const loadPreview = (massmailId, _recipientId, _data) => (dispatch, getState) => {
  const state = getState();
  const recipientId = _recipientId || getFirstContantId(state);
  const data = _data || getPreviewArgs(state);

  return dispatch({
    type: 'PROMISE',
    actions: [aTypes.PREVIEW_REQUEST, aTypes.PREVIEW_RECEIVE, aTypes.PREVIEW_FAIL],
    promise: jsonApiCall({
      url: '/actions/massmail/savepreview',
      data: {
        recipientId,
        massmailId,
        ...data,
      },
    }).then((response) => {
      dispatch(previewStoreArgs(data));
      dispatch(previewStoreRecipientId(recipientId));

      return response;
    }),
  });
};

export const previewDestroy = () => ({ type: aTypes.PREVIEW_DESTROY });

export const saveMassmail = (massmailId, data) => (dispatch, getState) => {
  const handleSuccessSave = () => {
    dispatch(notificationsPush({ theme: 'success', text: 'Рассылка сохранена' }));
    dispatch(reload('MASS_MAIL_LIST'));
    dispatch(initialize('MASSMAIL', getFormValues('MASSMAIL')(getState())));
  };

  if (data.send) {
    dispatch(loadPreview(massmailId, null, data)).then(handleSuccessSave);
    return null;
  }

  const promise = dispatch({
    type: 'PROMISE',
    actions: [
      aTypes.MASSMAIL_SAVE_REQUEST,
      aTypes.MASSMAIL_SAVE_RECEIVE,
      aTypes.MASSMAIL_SAVE_FAIL,
    ],
    promise: jsonApiCall({
      url: '/actions/massmail/save',
      data: {
        massmailId,
        ...data,
      },
    }),
  });

  promise.then(handleSuccessSave);

  return promise;
};

export const removeMassmail = (massmailId, callback) => (dispatch) => {
  if (global.confirm('Удалить рассылку?')) {
    const promise = jsonApiCall({
      url: '/actions/massmail/delete',
      data: { massmailId },
    });

    promise
      .then(() => {
        dispatch(reload('MASS_MAIL_LIST'));
        return null;
      })
      .then(callback);

    dispatch({
      type: 'PROMISE',
      actions: [
        aTypes.MASSMAIL_REMOVE_REQUEST,
        aTypes.MASSMAIL_REMOVE_RECEIVE,
        aTypes.MASSMAIL_REMOVE_FAIL,
      ],
      promise,
    });
  }
};

export const destroy = () => ({ type: aTypes.MASSMAIL_DESTROY });

export const cloneMassmail = (massmailId) => (dispatch) => {
  jsonApiCall({
    url: '/view/massmail/clone',
    data: { massmailId },
  }).then(({ clonedMassmailId }) => {
    dispatch(reload('MASS_MAIL_LIST'));
    history.push(`${ROUTER_PATH}/${clonedMassmailId}`);
  });
};
