import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import { config } from 'services/Config';
import Checkbox from '@crm/components/dist/lego2/Checkbox';
import Menu from './HeaderMenu';
import Filter from './Filter';
import TotalContacts from './TotalContacts';
import List from './List';
import Warning from './Warning';
import css from './Accounts.modules.scss';

const WARNINGS = {
  allowSend: [
    {
      type: 'Error',
      message: 'Ошибка, отправка невозможна',
    },
  ],
  maxCountExceeded: [
    {
      type: 'Error',
      message: 'Количество получателей не должно быть больше 300',
    },
  ],
};

const MassmailAccountsLayout = (props) => {
  const {
    className,
    items,
    recipientCount,
    recipientFilter,
    canDuplicate,
    refresh,
    changeFilter,
    onAccount,
    onAccounts,
    onFile,
    onClone,
    accountProps,
    contactProps,
    readonly,
    allowClone,
    allowSend,
    maxCountExceeded,
    onDuplicateChange,
    generalAccountMail,
    onGeneralAccountMailChange,
    ...rest
  } = props;

  return (
    <div className={cx(className, css.root)} {...rest}>
      <div className={css.header}>
        <div className={css.title}>Аккаунты</div>
        {!readonly && (
          <a className={css.refresh} tabIndex="0" onClick={refresh}>
            Обновить список
          </a>
        )}
        {config.value.features.massMailImprovements && allowClone && (
          <a className={css.refresh} tabIndex="1" onClick={onClone}>
            Повторить рассылку
          </a>
        )}
      </div>
      {!readonly && (
        <Menu className={css.menu} onAccount={onAccount} onAccounts={onAccounts} onFile={onFile} />
      )}
      <Filter
        className={css.select}
        onChange={changeFilter}
        filter={recipientFilter}
        readonly={readonly}
      />
      {config.value.features.massMailImprovements && !readonly && (
        <div className={css.info}>
          <Checkbox
            checked={generalAccountMail}
            onChange={() => onGeneralAccountMailChange(!generalAccountMail)}
          >
            Общее письмо для аккаунта
          </Checkbox>
        </div>
      )}
      {config.value.features.massMailImprovements && !readonly && (
        <div className={css.info}>
          <Checkbox checked={canDuplicate} onChange={() => onDuplicateChange(!canDuplicate)}>
            Разрешить повторение адресатов
          </Checkbox>
        </div>
      )}
      <div className={css.info}>
        <TotalContacts className={css.total} total={recipientCount} />
        {!allowSend && (
          <Warning
            className={css.error}
            totalType="Error"
            items={maxCountExceeded ? WARNINGS.maxCountExceeded : WARNINGS.allowSend}
          />
        )}
      </div>
      <List
        className={css.list}
        accountProps={accountProps}
        contactProps={contactProps}
        items={items}
        readonly={readonly}
      />
    </div>
  );
};

MassmailAccountsLayout.propTypes = {
  className: PropTypes.string,
  items: PropTypes.arrayOf(PropTypes.instanceOf(Object)),
  recipientCount: PropTypes.number,
  recipientFilter: PropTypes.objectOf(PropTypes.any),
  refresh: PropTypes.func,
  changeFilter: PropTypes.func,
  onAccount: PropTypes.func,
  onAccounts: PropTypes.func,
  onFile: PropTypes.func,
  readonly: PropTypes.bool,
  allowClone: PropTypes.bool,
  allowSend: PropTypes.bool,
  maxCountExceeded: PropTypes.bool,
};

MassmailAccountsLayout.defaultProps = {
  className: undefined,
  items: [],
  recipientCount: null,
  recipientFilter: {},
  refresh: () => {},
  changeFilter: () => {},
  onAccount: () => {},
  onAccounts: () => {},
  onFile: () => {},
  onClone: () => {},
  readonly: false,
  allowClone: false,
  allowSend: true,
  maxCountExceeded: false,
};

export default MassmailAccountsLayout;
