import React, { useCallback, useEffect, useState } from 'react';
import cx from 'classnames';
import { Calendar as CalendarComponent } from 'components/Calendar';
import { Event } from 'components/Calendar/Event/Event.types';
import { useDynamicallyModalForm } from 'components/DynamicallyModalForm';
import { fetchEvents } from '../../fetchEvents';
import { deleteEvent } from '../../deleteEvent';
import { useRefreshSubject, UpdateEvent } from '../../RefreshContext';
import css from './Calendar.module.css';
import { CalendarProps } from './Calendar.types';

const noNeedReaction = new Set(['ChangeDate']);

export const Calendar: React.FC<CalendarProps> = (props) => {
  const { date, onNavigate, view = 'day', className, eventMinHeight } = props;

  const [events, setEvents] = useState<Event[]>([]);
  const [visibleRange, setVisibleRange] = useState<[Date, Date]>();
  const refreshSubject = useRefreshSubject();
  const activityForm = useDynamicallyModalForm();

  const reloadCalendar = useCallback(
    (update?: UpdateEvent) => {
      if (!visibleRange || (update && noNeedReaction.has(update.type))) {
        return;
      }
      return fetchEvents(visibleRange).then((events) => {
        setEvents(events);
      });
    },
    [visibleRange],
  );

  const handleRangeChange = (range: [Date, Date]) => {
    setVisibleRange(range);
  };

  const handleSuccessSubmit = (response) => {
    refreshSubject.next({ type: 'UpdateEvent', payload: response });
  };

  const handleEventEdit = async (event) => {
    if (event.editingUrl) {
      activityForm.setUrl(event.editingUrl).loadForm();
      activityForm.setSuccessSubmitListener(handleSuccessSubmit);
    }
  };

  const handleEventDelete = (event) => {
    deleteEvent(event).then(handleSuccessSubmit);
  };

  useEffect(() => {
    const sub = refreshSubject.subscribe(reloadCalendar);

    return () => {
      sub.unsubscribe();
    };
  }, [reloadCalendar, refreshSubject]);

  useEffect(() => {
    const promise = reloadCalendar();
    return () => {
      promise?.cancel();
    };
  }, [visibleRange, reloadCalendar]);

  return (
    <CalendarComponent
      date={date}
      view={view}
      className={cx(css.Calendar, className)}
      events={events}
      onNavigate={onNavigate}
      onRangeChange={handleRangeChange}
      onEventEdit={handleEventEdit}
      onEventDelete={handleEventDelete}
      eventMinHeight={eventMinHeight}
      toolbar={false}
    />
  );
};
