import React, { FC, useCallback, useEffect, useMemo } from 'react';
import { observer } from 'mobx-react-lite';
import cx from 'classnames';
import { ThemeProvider } from 'components/Table/themes';
import { withPagination } from 'components/Table/hoc';
import { TableEditable } from 'components/Table';
import { Provider, TableController } from 'components/Table/services/TableController';
import { useRefreshSubject } from '../../RefreshContext';
import { PaginationPage } from '../../../../types/api/form/Form';
import css from './Main.module.css';
import { MainProps } from './Main.types';
import { TABLE_SOURCE_URL } from './Main.config';

const Table = withPagination(TableEditable);

export const Main: FC<MainProps> = observer(
  ({ url = TABLE_SOURCE_URL, withoutPadding = false }) => {
    const refreshSubject = useRefreshSubject();
    const controller = useMemo<TableController>(() => new TableController(), []);

    const loadTableData = useCallback(() => {
      controller.fetch();
    }, [controller]);

    useEffect(() => {
      if (url) {
        controller.setBaseUrl(url);
      }
    }, [controller, url]);

    useEffect(() => {
      if (refreshSubject) {
        const sub = refreshSubject.subscribe(loadTableData);

        return () => sub.unsubscribe();
      }
    }, [refreshSubject, loadTableData]);

    const handlePaginationItemClick = (page: PaginationPage) => {
      controller.setCurrentPage(Number(page.caption)).fetch();
    };

    const updateTableAction = () => {
      if (refreshSubject) {
        refreshSubject.next({ type: 'UpdateTable' });
      }
      loadTableData();
    };

    useEffect(() => {
      loadTableData();
    }, [loadTableData]);

    if (controller.errorMessage) {
      return <span>{controller.errorMessage}</span>;
    }

    if (!controller.tableData) {
      return null;
    }

    return (
      <div className={cx(css.Main, { [css.Main__withoutPadding]: withoutPadding })}>
        <Provider value={controller}>
          <ThemeProvider value="neo">
            <Table
              withControls
              updateTableAction={updateTableAction}
              tableData={controller.tableData}
              pagination={controller.tableData.gridPagination}
              paginationItemClick={handlePaginationItemClick}
            />
          </ThemeProvider>
        </Provider>
      </div>
    );
  },
);
