import React, { useState, useEffect } from 'react';
import cx from 'classnames';
import { config } from 'services/Config';
import { get } from 'api/common';
import { ActionsPanel } from 'components/ActionsPanel';
import Icon from '@crm/components/dist/lego2/Icon';
import { useDynamicallyModalForm } from 'components/DynamicallyModalForm';
import css from './ActivityDropdown.module.css';
import { useRefreshSubject } from '../../RefreshContext';
import { FormAction } from '../../../../types/api/form/Form';
import { ACTIVITIES_LIST_URL } from './ActivityDropdown.constants';

export const ActivityDropdown = () => {
  const [activities, setActivities] = useState([]);
  const activityForm = useDynamicallyModalForm();
  const refreshSubject = useRefreshSubject();
  useEffect(() => {
    const url = ACTIVITIES_LIST_URL;
    const request = get({
      url,
    }).then(({ actions }) => {
      setActivities(actions);
    });
    return () => request.cancel();
  }, []);

  if (activities.length === 0) {
    return null;
  }

  const handleSuccessSubmit = (response) => {
    refreshSubject.next({ type: 'CreateActivity', payload: response });
  };

  const handleDropdownChange = (action: FormAction) => {
    const { actionUrl: url } = action;
    if (url) {
      activityForm.setUrl(url).loadForm();
      activityForm.setSuccessSubmitListener(handleSuccessSubmit);
    }
  };

  const isNewDesign = config.value.features.newActivitiesDesign;
  const iconProvider = (cls) => <Icon className={cls} svg="plus" />;

  return (
    <ActionsPanel
      direction="vertical"
      onItemClick={handleDropdownChange}
      view={isNewDesign ? 'action' : 'pseudo'}
      iconLeft={isNewDesign ? iconProvider : undefined}
      menuClassName={css.ActivityDropdown_menu}
      actions={activities}
      className={cx(css.ActivityDropdown, {
        [css.ActivityDropdown_newDesign]: isNewDesign,
      })}
    />
  );
};
