import React, { FC, useState, memo, useCallback, useEffect } from 'react';
import cx from 'classnames';
import { DatePicker } from 'components/DatePicker';
import { isPast, isToday } from 'date-fns';
import css from './Sidebar.module.css';
import { SidebarProps } from './Sidebar.types';
import { fetchDates, createDateKey } from './Sidebar.utils';
import { useRefreshSubject } from '../RefreshContext';
import { ActivityDropdown } from './ActivityDropdown';

const noNeedReaction = new Set(['ChangeDate']);

export const Sidebar: FC<SidebarProps> = memo((props) => {
  const { date, onDateChange, afterActivities } = props;
  const [visibleRange, setVisibleRange] = useState<[Date, Date]>();
  const [expiredDatesMap, setExpiredDatesMap] = useState<Record<string, boolean>>({});
  const refreshSubject = useRefreshSubject();

  const reloadDatePicker = useCallback(
    (update?: { type: string }) => {
      if (!visibleRange || (update && noNeedReaction.has(update.type))) {
        return;
      }

      return fetchDates(visibleRange).then((response) => {
        const newMap = response.dates.reduce<Record<string, boolean>>(
          (acc, date) => ({
            ...acc,
            [createDateKey(date.date)]: date.hasExpiredActivities,
          }),
          {},
        );
        setExpiredDatesMap((prev) => ({
          ...prev,
          ...newMap,
        }));
      });
    },
    [visibleRange],
  );

  const handleChange = (date: Date | null) => {
    if (date) {
      onDateChange(date);
    }
  };

  const handleVisibleRange = (range: [Date, Date]) => {
    setVisibleRange(range);
  };

  const dayClassName = (dayDate: Date): string | null => {
    let resultClassName = '';
    if (expiredDatesMap[createDateKey(dayDate)]) {
      resultClassName = cx(resultClassName, css.Sidebar__dayHasExpiredTodos);
    }

    if (isPast(dayDate) && !isToday(dayDate)) {
      resultClassName = cx(resultClassName, css.Sidebar__pastDay);
    }

    return resultClassName;
  };

  useEffect(() => {
    const promise = reloadDatePicker();
    return () => {
      promise?.cancel();
    };
  }, [visibleRange, reloadDatePicker]);

  useEffect(() => {
    const sub = refreshSubject.subscribe(reloadDatePicker);
    return () => {
      sub.unsubscribe();
    };
  }, [refreshSubject, reloadDatePicker]);

  return (
    <div className={css.Sidebar}>
      <h2 className={css.Sidebar__title}>Фильтры</h2>
      <ActivityDropdown />
      {afterActivities}
      <DatePicker
        selected={date}
        onChange={handleChange}
        dayClassName={dayClassName}
        onVisibleRange={handleVisibleRange}
        calendarClassName={css.Sidebar__datepicker}
      />
    </div>
  );
});
