import React, { FC, useState, useEffect, useCallback } from 'react';
import cx from 'classnames';
import { observer } from 'mobx-react-lite';
import { PaginationPage } from 'types/api/form/Form';
import { TableEditable } from 'components/Table/TableEditable';
import { ThemeProvider } from 'components/Table/themes';
import { TableController, Provider } from 'components/Table/services/TableController';
import { withPagination } from 'components/Table/hoc';
import css from './TodoTable.module.css';
import { TableWithFiltersProps } from './TodoTable.types';
import { UpdateEvent, useRefreshSubject } from '../../RefreshContext';

const Table = withPagination(TableEditable);
const noNeedReaction = new Set(['UpdateTable']);

export const TodoTable: FC<TableWithFiltersProps> = observer(
  ({ date, url, title, extendedQueryParams, className }) => {
    const [controller] = useState<TableController>(() => new TableController());
    const refreshSubject = useRefreshSubject();

    const reloadTable = useCallback(
      (update?: UpdateEvent) => {
        if (update && noNeedReaction.has(update?.type)) {
          return;
        }
        controller.fetch();
      },
      [controller],
    );

    useEffect(() => {
      if (url) {
        controller.setBaseUrl(url);
      }
    }, []);

    useEffect(() => {
      controller.setExtendedQueryParams(extendedQueryParams);
    }, [extendedQueryParams]);

    useEffect(() => {
      if (refreshSubject) {
        const sub = refreshSubject.subscribe(reloadTable);
        return () => sub.unsubscribe();
      }
    }, [refreshSubject, reloadTable]);

    useEffect(() => {
      reloadTable();
    }, [date]);

    if (controller.errorMessage) {
      return <span>{controller.errorMessage}</span>;
    }

    const { tableData } = controller;
    if (!tableData) {
      return null;
    }

    const handlePaginationItemClick = (page: PaginationPage) => {
      controller.setCurrentPage(Number(page.caption)).fetch();
    };

    const updateTableAction = () => {
      if (refreshSubject) {
        refreshSubject.next({ type: 'UpdateTable' });
      }
      controller.fetch();
    };

    return (
      <Provider value={controller}>
        <div className={cx(css.TodoTable, className)}>
          <ThemeProvider value="neo">
            <Table
              title={title}
              withControls
              updateTableAction={updateTableAction}
              tableData={tableData}
              pagination={tableData.gridPagination}
              paginationItemClick={handlePaginationItemClick}
            />
          </ThemeProvider>
        </div>
      </Provider>
    );
  },
);
