import React from 'react';
import { Field, FieldArray, reduxForm, InjectedFormProps } from 'redux-form';
import { FormattedRelativeTime } from 'react-intl';
import cx from 'classnames';
import { EType } from 'types/entities';
import { buildUrl } from 'utils/buildUrl';
import Prompt from 'components/Prompt';
import Button, { ButtonPinProp } from '@crm/components/dist/lego2/Button';
import { ImperativePopup, ImperativePopupProps } from '@crm/components/dist/lego2/Popup';
import Select2 from '@crm/components/dist/lego2/Select';
import FieldEmail from 'requestForms/components/FieldEmail';
import AttachFiles from 'lego/redux-form/AttachFiles';
import Templates from 'modules/templates/components/Templates';
import { BackendFile } from 'lego/components/AttachFiles/types';
import { MediaMatchContext, DeviceType } from 'components/MediaMatch';
import { withAsyncFieldProvider, AsyncFormContext } from 'lego/hoc/asyncField';
import createI18N, { i18nRaw as createI18NRaw } from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import SuggestGrid from 'lego/redux-form/SuggestGrid';
import { Account } from 'types';
import PinGroup from 'lego/components/PinGroup';
import Icon from 'lego/components/Icon';
import Toolbar from 'components/Toolbar';
import Spinner from 'components/Spinner';
import DateTimePicker from 'lego/redux-form/DateTimePicker';
import DropdownMenu from 'lego/components/DropdownMenu';
import RichTextEditor from 'lego/redux-form/RichTextEditor';
import Select from 'lego/redux-form/Select';
import TextInput from 'lego/redux-form/TextInput';
import { TagsField as Tags, apiMail as apiTags } from 'modules/tags';
import { FieldImportant } from 'modules/mailIndicators';
import CheckBox from 'lego/redux-form/CheckBox';
import * as keyset from '../newMail.i18n';
import css from './styles.modules.scss';
import rowCss from './row.modules.scss';
import Signature from './Signature';
import Macros from './Macros';
import DelayedDate from './DelayedDate';
import filterValuesForBackend from '../filterValuesForBackend';
import CustomField, { Props as CustomFieldProps } from './CustomField';
import { CustomSendButton } from './types';

import { Signature as ISignature, Template } from '../newMailSlice';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nRaw = createI18NRaw(keyset);
const i18nSend = commonI18n('send');
const i18nSaveAsDraft = i18n('saveAsDraft');
const i18nCancel = commonI18n('cancel');
const i18nFrom = commonI18n('sendFrom');
const i18nTo = commonI18n('sendTo');
const i18nACopy = commonI18n('aCopy');
const i18nBcc = commonI18n('bcc');
const i18nSubject = commonI18n('subject');
const i18nRemoveTimer = i18n('removeTimer');
const i18nPhoneTextCancel = i18n('phoneTextCancel');

export interface Fields {
  important?: boolean;
  from?: boolean;
  macros?: boolean;
  to?: boolean;
  cc?: boolean;
  bcc?: boolean;
  subject?: boolean;
  tags?: boolean;
  files?: boolean;
  delayedDate?: boolean;
  opportunities?: boolean;
}

export interface Props {
  name: string;
  loadTemplates: () => void;
  onCancel: () => void;
  onSubmit: (values: object, autoSave: boolean) => void;

  // optional, but there're defaultProps
  froms: string[];
  customFieldsTop: CustomFieldProps[];
  customFieldsBottom: CustomFieldProps[];
  EmailInput: React.ComponentType;
  fields: Fields;
  buttonTextSend: React.ReactNode;
  buttonTextSave: React.ReactNode;
  buttonTextCancel: React.ReactNode;
  objectName: string;
  disabledSend: boolean;
  disabledAll: boolean;
  isLoad: boolean;
  clearHtmlOnSend: boolean;
  showLabels: boolean;
  showPlaceholders: boolean;
  loadOnMount: boolean;
  isShowCancel: boolean;
  isSwitchSaveButtons: boolean;
  isEscapeCancel: boolean;
  isPreventUnload: boolean;
  focusOnEditor: boolean;

  // fully optional
  account?: Account;
  noLinkWithOpportunity?: boolean;
  customSendButtons?: CustomSendButton[];
  className?: string;
  resizeHash?: string;
  classNameFooter?: string;
  classNameBody?: string;
  type?: string;
  style?: React.CSSProperties;
  objId?: string | number;
  replyToMailId?: string | number;
  delayedDate?: string;
  autoSaveDate?: string;
  files?: {
    [key: string]: BackendFile;
  };
  macros?: string[];
  templates?: {
    signatures: ISignature[];
    templates: Template[];
  };
  load?: () => void;
  destroyNewMail?: () => void;
  onSave?: () => void;
  setAutoSaveFunc?: () => void;
  renderExtraActions?: () => void;
  resetOpportunities?: () => void;
}

interface State {
  ckeditor?: {
    mode: string;
    getData: () => string;
    setData: (data: string) => void;
    insertHtml: (value: string) => void;
  };
}

const defaultProps = {
  froms: [],
  customFieldsTop: [],
  customFieldsBottom: [],
  EmailInput: FieldEmail,
  fields: {},
  buttonTextSend: i18nSend,
  buttonTextSave: i18nSaveAsDraft,
  buttonTextCancel: i18nCancel,
  objectName: 'mail',
  submitting: false,
  disabledSend: false,
  disabledAll: false,
  isLoad: false,
  clearHtmlOnSend: true,
  showLabels: true,
  showPlaceholders: false,
  loadOnMount: false,
  isShowCancel: true,
  isSwitchSaveButtons: false,
  isEscapeCancel: true,
  isPreventUnload: false,
  pristine: false,
  focusOnEditor: true,
};

const dropdownMenuItems = [
  {
    val: 'saveAsDraft',
    text: i18nSaveAsDraft,
  },
  {
    val: 'close',
    text: i18nPhoneTextCancel,
  },
];

class NewMail extends React.Component<Props & InjectedFormProps, State> {
  public static contextType = MediaMatchContext;

  private filesRef: React.RefObject<{
    updateFiles: (ids: number[]) => void;
  }> = React.createRef();

  public static defaultProps = defaultProps;

  private autoSave: () => void;

  private handleSend: () => void;

  private handleSave: () => void;

  private form: React.RefObject<HTMLFormElement> = React.createRef();

  public constructor(props) {
    super(props);

    this.state = { ckeditor: undefined };

    this.autoSave = this.props.handleSubmit(this.submit(false, true));
    this.handleSend = this.props.handleSubmit(this.submit(true));
    this.handleSave = this.props.handleSubmit(this.submit(false, true, this.props.onSave));

    if (typeof props.setAutoSaveFunc === 'function') {
      props.setAutoSaveFunc(this.autoSave);
    }
  }

  public componentDidMount() {
    const { loadTemplates, loadOnMount, load } = this.props;
    loadTemplates();
    if (loadOnMount && load) {
      load();
    }
  }

  public componentWillUnmount() {
    const { destroyNewMail } = this.props;
    if (destroyNewMail) {
      destroyNewMail();
    }
  }

  private isPlain = (): boolean => {
    if (this.state.ckeditor && this.state.ckeditor.mode === 'source') {
      return true;
    }

    return false;
  };

  private handleDropdownMenuClick = (_, value) => {
    if (value === 'saveAsDraft') {
      this.handleSave();
    }

    if (value === 'close' && this.props.onCancel) {
      this.props.onCancel();
    }
  };

  private handleKeyUp = (event: React.KeyboardEvent): void => {
    const { onCancel, isEscapeCancel } = this.props;

    if (isEscapeCancel && event.key === 'Escape' && typeof onCancel === 'function') {
      onCancel();
    }
  };

  private handleCustomSubmitButton = (action: string) => (resolutionId: string) => {
    return this.props.handleSubmit((values: object) =>
      this.props.onSubmit(
        filterValuesForBackend(
          // @ts-ignore
          { ...values, resolutionId, action, send: true },
          this.props,
          this.isPlain(),
        ),
        false,
      ),
    )();
  };

  private submit = (
    send: boolean | undefined = undefined,
    autoSave: boolean | undefined = false,
    func = this.props.onSubmit,
  ) => (values) => {
    return func(filterValuesForBackend({ ...values, send }, this.props, this.isPlain()), autoSave);
  };

  private handleTemplateClick = (template: Template): void => {
    if (this.filesRef.current && Array.isArray(template.files)) {
      const fileIds = template.files.map((file) => file.id);
      if (fileIds.length) {
        this.filesRef.current.updateFiles(fileIds);
      }
    }
  };

  // eslint-disable-next-line class-methods-use-this
  private handleSubmit(event: React.FormEvent): void {
    event.preventDefault();
  }

  private renderImperativePopupContent = (): React.ReactNode => (
    <Field
      name="delayedDate"
      component={DateTimePicker}
      initOnMount
      initOffset={{
        days: 1,
      }}
      view="inline"
      minDate={new Date()}
      time
      autoFocus
    />
  );

  private imperativePopupIconProvider = (iconCls): React.ReactElement => (
    <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="access_time" />
  );

  private renderImperativePopupToggle = (_, popupProps): React.ReactNode => {
    return (
      <Button
        className={popupProps.className}
        view="pseudo"
        disabled={this.props.submitting}
        tabIndex={-1}
        pin={(popupProps as ImperativePopupProps & { pin: ButtonPinProp }).pin}
        icon={this.imperativePopupIconProvider}
      />
    );
  };

  private onDelayedDateClick = () => {
    this.props.change('delayedDate', '');
  };

  private delayedDateIconProvider = (iconCls) => (
    <Icon pack={Icon.PACKS.MATERIAL} icon="delete" className={iconCls} />
  );

  private setCkeditor = (ckeditor) => {
    this.setState({ ckeditor });
  };

  public renderToolbar = (async: boolean): React.ReactNode => {
    const {
      fields,
      disabledSend,
      disabledAll,
      buttonTextSend,
      buttonTextSave,
      buttonTextCancel,
      isShowCancel,
      renderExtraActions,
      isSwitchSaveButtons,
      submitting,
      customSendButtons,
    } = this.props;

    const deviceType = this.context;

    const sendButton = (
      <span>
        <PinGroup>
          <Button
            onClick={this.handleSend}
            disabled={async || disabledSend || disabledAll || submitting}
            tabIndex={-1}
            view={isSwitchSaveButtons ? 'pseudo' : 'action'}
          >
            {buttonTextSend}
            {this.props.delayedDate && (
              <span>
                &nbsp;
                <DelayedDate delayedDate={this.props.delayedDate} />
              </span>
            )}
          </Button>
          {fields.delayedDate && (
            <ImperativePopup
              directions={['top-center']}
              content={this.renderImperativePopupContent}
              toggle={this.renderImperativePopupToggle}
              padding="xs"
            />
          )}
          {fields.delayedDate && this.props.delayedDate && (
            <Button
              view="danger"
              title={i18nRemoveTimer}
              disabled={disabledAll || submitting}
              onClick={this.onDelayedDateClick}
              icon={this.delayedDateIconProvider}
              tabIndex={-1}
            />
          )}
        </PinGroup>
      </span>
    );

    const phone = (
      <>
        {sendButton}
        <DropdownMenu
          arrow={false}
          theme="pseudo"
          text="..."
          directions={['top', 'top-right', 'top-left']}
          items={dropdownMenuItems}
          onClick={this.handleDropdownMenuClick}
        />
      </>
    );

    const desktop = (
      <>
        {sendButton}
        {Array.isArray(customSendButtons) &&
          customSendButtons.map(({ value, text, options }) => (
            <Select2
              key={value}
              view="default"
              onChange={this.handleCustomSubmitButton(value)}
              hasEmptyValue={false}
              tabIndex={-1}
              disabled={async || disabledSend || disabledAll || submitting}
              placeholder={text}
              options={options}
            />
          ))}
        {!isSwitchSaveButtons && (
          <Button
            view="pseudo"
            onClick={this.handleSave}
            tabIndex={-1}
            disabled={async || disabledAll || submitting}
          >
            {buttonTextSave}
          </Button>
        )}
        {isShowCancel && (
          <Button
            view="pseudo"
            onClick={this.props.onCancel}
            disabled={disabledAll || submitting}
            tabIndex={-1}
          >
            {buttonTextCancel}
          </Button>
        )}
        {typeof renderExtraActions === 'function' && renderExtraActions()}
      </>
    );

    return (
      <Toolbar className={css.toolbar}>
        {isSwitchSaveButtons && (
          <Button
            onClick={this.handleSave}
            tabIndex={-1}
            disabled={async || disabledAll || submitting}
            view="action"
          >
            {buttonTextSave}
          </Button>
        )}
        {deviceType === DeviceType.Phone ? phone : desktop}
      </Toolbar>
    );
  };

  public render() {
    const {
      className,
      classNameFooter,
      classNameBody,
      customFieldsTop,
      customFieldsBottom,
      isLoad,
      fields,
      style,
      showLabels,
      showPlaceholders,
      autoSaveDate,
      isPreventUnload,
      focusOnEditor,
      pristine,
      type,
      account,
      noLinkWithOpportunity,
      resetOpportunities,
    } = this.props;
    if (!isLoad) {
      return <Spinner full visible />;
    }

    return (
      <form
        onSubmit={this.handleSubmit}
        onKeyUp={this.handleKeyUp}
        role="presentation"
        className={cx(css.root, fields.important && css.root_important, className)}
        style={style}
        ref={this.form}
      >
        <Prompt when={isPreventUnload && !pristine} />
        {fields.important && (
          <Field
            className={css.important}
            name="importance"
            type="checkbox"
            component={FieldImportant}
          />
        )}
        <div className={css.header}>
          <div className={cx(rowCss.row, rowCss.row_topButtons)}>
            {fields.from && showLabels && <span className={rowCss.label}>{i18nFrom}</span>}
            <div className={cx(rowCss.rowFields, rowCss.rowFields_topButtons)}>
              {fields.from && (
                <Field
                  className={rowCss.field}
                  componentClassName={rowCss.input}
                  width="max"
                  size="s"
                  name="from"
                  id="from"
                  component={Select}
                  hasEmptyValue={false}
                  options={this.props.froms.map((item, i) => ({
                    value: i.toString(),
                    content: item,
                  }))}
                />
              )}
              {fields.macros && (
                <Macros macros={this.props.macros} ckeditor={this.state.ckeditor} />
              )}
              <Signature
                signatures={this.props.templates && this.props.templates.signatures}
                ckeditor={this.state.ckeditor}
                isUpdateOnInit={type !== 'draft'}
              />
              <Templates
                container={this.form}
                templates={this.props.templates && this.props.templates.templates}
                ckeditor={this.state.ckeditor}
                onTemplateClick={this.handleTemplateClick}
              />
            </div>
          </div>
          {customFieldsTop.map((field) => (
            <CustomField
              key={field.name}
              {...field}
              label={showLabels ? field.label : undefined}
              theme={rowCss}
              showPlaceholder={showPlaceholders}
            />
          ))}
          {fields.to && (
            <div className={rowCss.row}>
              {showLabels && <span className={rowCss.label}>{i18nTo}</span>}
              <Field
                size="m"
                className={rowCss.field}
                componentClassName={rowCss.input}
                name="to"
                id="to"
                component={this.props.EmailInput}
                type="text"
                placeholder={showPlaceholders ? i18nTo : undefined}
              />
            </div>
          )}
          {fields.cc && (
            <div className={rowCss.row}>
              {showLabels && <span className={rowCss.label}>{i18nACopy}</span>}
              <Field
                size="m"
                className={rowCss.field}
                componentClassName={rowCss.input}
                name="cc"
                id="cc"
                component={this.props.EmailInput}
                type="text"
                placeholder={showPlaceholders ? i18nACopy : undefined}
              />
            </div>
          )}
          {fields.bcc && (
            <div className={rowCss.row}>
              {showLabels && <span className={rowCss.label}>{i18nBcc}</span>}
              <Field
                size="m"
                className={rowCss.field}
                componentClassName={rowCss.input}
                name="bcc"
                id="bcc"
                component={this.props.EmailInput}
                type="text"
                placeholder={showPlaceholders ? i18nBcc : undefined}
              />
            </div>
          )}
          {fields.subject && (
            <div className={rowCss.row}>
              {showLabels && <span className={rowCss.label}>{i18nSubject}</span>}
              <Field
                size="s"
                className={rowCss.field}
                componentClassName={rowCss.input}
                name="subject"
                id="subject"
                component={TextInput}
                type="text"
                placeholder={showPlaceholders ? i18nSubject : undefined}
                emptyWarning
                warnings={{
                  notEmpty: true,
                }}
              />
            </div>
          )}

          {fields.opportunities && (
            <>
              {account?.id && !noLinkWithOpportunity && (
                <div className={rowCss.row}>
                  {showLabels && <span className={rowCss.label}>Сделки</span>}
                  <Field
                    className={rowCss.field}
                    componentClassName={rowCss.input}
                    name="opportunities"
                    component={SuggestGrid}
                    provider={buildUrl('/v0/blocks/opportunity/suggest', {
                      accountId: account.id,
                      eType: EType.Mail,
                      eId: this.props.objId || this.props.replyToMailId,
                    })}
                    account={account}
                    title="Привяжите сделки"
                  />
                </div>
              )}
              <Field
                className={css.checkbox}
                name="noLinkWithOpportunity"
                component={CheckBox}
                onChange={resetOpportunities}
              >
                Письмо не относится к сделке
              </Field>
            </>
          )}

          {customFieldsBottom.map((field) => (
            <CustomField
              key={field.name}
              {...field}
              label={showLabels ? field.label : undefined}
              theme={rowCss}
              showPlaceholder={showPlaceholders}
            />
          ))}
          {fields.tags && (
            <FieldArray name="TagIds" component={Tags} className={css.tags} getTags={apiTags.get} />
          )}
          {fields.files && (
            <Field
              name="files"
              component={AttachFiles}
              className={css.files}
              files={this.props.files}
              objectName={this.props.objectName}
              objId={this.props.objId}
              refComponent={this.filesRef}
            />
          )}
        </div>
        <Field
          name="body"
          component={RichTextEditor}
          resizeHash={this.props.resizeHash}
          className={cx(css.body, classNameBody)}
          config="mail"
          type="text"
          onInstanceReady={this.setCkeditor}
          focus={focusOnEditor}
          height="auto"
        />
        <div className={cx(css.footer, classNameFooter)}>
          <AsyncFormContext.Consumer>{this.renderToolbar}</AsyncFormContext.Consumer>
          {autoSaveDate && (
            <span className={css.autoSave}>
              {i18nRaw('autoSave', {
                time: (
                  <FormattedRelativeTime
                    numeric="auto"
                    updateIntervalInSeconds={10}
                    value={0}
                    key={autoSaveDate}
                  />
                ),
              })}
            </span>
          )}
        </div>
      </form>
    );
  }
}

export default reduxForm({})(
  withAsyncFieldProvider()(NewMail) as React.ComponentType<Props & InjectedFormProps>,
);
