import * as React from 'react';
import DropdownMenu from 'lego/components/DropdownMenu';
import createI18N from '@yandex-int/i18n';
import * as keyset from './Signature.i18n';
import SignatureEditor from '../../utils/SignatureEditor';
import css from './styles.modules.scss';

const i18nSignature = createI18N(keyset)('signature');

interface Props {
  ckeditor: any;
  signatures: any;
  isUpdateOnInit: boolean;
}

class Signature extends React.Component<Props> {
  private signatureEditor: SignatureEditor;

  public static defaultProps = {
    signatures: [],
    isUpdateOnInit: true,
  };

  public constructor(props) {
    super(props);
    this.signatureEditor = new SignatureEditor(props.signatures);
  }

  public onChange = (e, signatureId: number) => {
    const { ckeditor } = this.props;

    if (ckeditor) {
      const isHtml = ckeditor.mode !== 'source';

      const updatedContent = this.signatureEditor.updateSignature(
        ckeditor.getData(),
        ckeditor.mode !== 'source',
        signatureId,
      );

      if (updatedContent) {
        ckeditor.setData(updatedContent);
        return;
      }

      if (isHtml) {
        const signatureTemplate = this.signatureEditor.getSignatureHtmlTemplate(signatureId);

        ckeditor.insertHtml(signatureTemplate);

        if (signatureTemplate) {
          const range = ckeditor.createRange();
          range.moveToElementEditStart(range.root);
          ckeditor.getSelection().selectRanges([range]);
        }
      }
    }
  };

  public render() {
    const { signatures } = this.props;

    const items = signatures.map(signature => ({
      val: signature.signatureId,
      text: (
        <span className={css.signature}>
          {signature.name != null ? (
            signature.name
          ) : (
            <>
              <span dangerouslySetInnerHTML={{ __html: signature.bodyHtml }} />
              <span className={css.signature__cover} />
            </>
          )}
        </span>
      ),
    }));

    return (
      <DropdownMenu
        onClick={this.onChange}
        size="s"
        disabled={!this.props.ckeditor}
        text={i18nSignature}
        items={items}
        maxHeight={400}
        popupClassName={css.popup}
      />
    );
  }
}

export default Signature;
