import * as React from 'react';
import { connect } from 'react-redux';
import { isSubmitting } from 'redux-form';
import getDisplayName from 'utils/getDisplayName';

export interface Props {
  onSubmit: () => void;
  form: string;

  isAutoSave: boolean;
  isSubmitting: boolean;
  autoSaveDelay: number;

  autoSaveSuccess?: (func: () => void) => void;
  onSubmitSuccess?: (data: any, ...args: any) => void;
}

interface WrapComponentProps {
  isSubmitting: boolean;
  onSubmit: () => void;

  setAutoSaveFunc?: (func: () => void) => void;
  onSubmitSuccess?: (data: any, ...args: any) => void;
}

const withAutoSave = (config?: Props) => (WrapComponent: React.ReactType<WrapComponentProps>) => {
  class AutoSave extends React.Component<Props> {
    public static defaultProps = {
      isAutoSave: true,
      isSubmitting: false,
      autoSaveSuccess: undefined,
      autoSaveDelay: 500,
      onSubmitSuccess: undefined,
      ...config,
    };

    public static displayName = `AutoSave(${getDisplayName(WrapComponent)})`;

    private static MS_IN_MINUTE = 1000 * 60;

    private interval: number;

    private autoSaveFunc: () => void;

    private isRunAutoSave: boolean;

    public componentDidMount(): void {
      if (this.props.isAutoSave) {
        this.interval = window.setInterval(this.runAutoSave, AutoSave.MS_IN_MINUTE);
      }
    }

    public componentWillUnmount(): void {
      window.clearInterval(this.interval);
    }

    private setAutoSaveFunc = (func: () => void): void => {
      this.autoSaveFunc = func;
    };

    private runAutoSave = (): void => {
      // eslint-disable-next-line no-shadow
      const { isSubmitting } = this.props;
      if (!isSubmitting && typeof this.autoSaveFunc === 'function') {
        this.isRunAutoSave = true;
        this.autoSaveFunc();
      }
    };

    private handleSubmitSuccess = (data, ...args): void => {
      if (this.isRunAutoSave) {
        this.isRunAutoSave = false;
        if (typeof this.props.autoSaveSuccess === 'function') {
          this.props.autoSaveSuccess(data);
        }
      } else if (typeof this.props.onSubmitSuccess === 'function') {
        this.props.onSubmitSuccess(data, ...args);
      }
    };

    public render() {
      const { isAutoSave, autoSaveDelay, ...rest } = this.props;
      const wrapComponentProps: WrapComponentProps = rest;

      if (isAutoSave) {
        wrapComponentProps.onSubmitSuccess = this.handleSubmitSuccess;
        wrapComponentProps.setAutoSaveFunc = this.setAutoSaveFunc;
      }

      return <WrapComponent {...wrapComponentProps} />;
    }
  }

  return connect<{}, {}, Props>((state, { form }) => ({
    isSubmitting: form && isSubmitting(form)(state),
  }))(AutoSave);
};

export default withAutoSave;
