/* eslint-disable @typescript-eslint/no-explicit-any */
import { connectAdvanced } from 'react-redux';
import isPromise from 'is-promise';
import shallowEqual from 'utils/shallowEqual';
import memoize from 'memoize-one';
import get from 'lodash/get';
import pick from 'lodash/pick';
import { formValueSelector, change } from 'redux-form';
import { File } from 'components/FilesList/types';
import { Tag } from 'types';
import { getState } from '../selectors';
import { actions as sliceActions } from '../newMailSlice';
import { Props as CustomFieldProps } from './CustomField';

const { loadMail, pushNotification, autoSaveSuccess } = sliceActions;

export interface Props {
  name: string;
  customFieldsTop: CustomFieldProps[];
  customFieldsBottom: CustomFieldProps[];
  loadOnMount: boolean;
  type: string;
  loadArgs: any;

  loadDraft: (args: any) => Promise<any>;
  loadReply: (args: any) => Promise<any>;
  loadNew: (args: any) => Promise<any>;

  onSubmit?: (values: any, autoSave?: boolean) => void;
  onCancel?: () => void;
}

interface InitialValues {
  body?: string;
  TagIds: Tag[] | number[];
  from: string;
  importance: boolean;
}

function selectorFactory(dispatch) {
  let props: Props = {} as Props;
  let resultProps: Props = {} as Props;

  type GetInitialValues = (data: {
    body?: string;
    froms: string[];
    isExternalRecipients: boolean;
    isHtml: boolean;
    isImportant: boolean;
  }) => InitialValues;

  const getInitialValues: GetInitialValues = memoize((data) => {
    const customFieldsTop = props.customFieldsTop || [];
    const customFieldsBottom = props.customFieldsBottom || [];

    return {
      ...pick(data, [
        'from',
        'body',
        'to',
        'cc',
        'bcc',
        'subject',
        'containerId',
        'replyToMailId',
        'mailId',
        'delayedDate',
        'files',
        'noLinkWithOpportunity',
        ...customFieldsTop.map((field) => field.name),
        ...customFieldsBottom.map((field) => field.name),
      ]),
      from: '0',
      TagIds: get(data, 'tags'),
      importance: get(data, 'isImportant'),
      opportunities: get(data, 'opportunities') || [],
    };
  });

  type GetStateProps = (
    state: any,
  ) => {
    froms: string[];
    enableReinitialize: boolean;
    isLoad: boolean;
    initialValues: InitialValues;

    objId?: number;
    files?: File[];
    autoSaveDate?: string;
    delayedDate?: string;
  };

  const getStateProps: GetStateProps = (state) => {
    const formDataSelector = formValueSelector(props.name);
    const mailState = getState(state, props.name) || {};

    return {
      objId: get(mailState, 'data.mailId') || formDataSelector(state, 'mailId'),
      replyToMailId: get(mailState, 'data.replyToMailId'),
      account: formDataSelector(state, 'account'),
      noLinkWithOpportunity: formDataSelector(state, 'noLinkWithOpportunity'),
      froms: get(mailState, 'data.froms'),
      files: get(mailState, 'data.files'),
      delayedDate: formDataSelector(state, 'delayedDate'),
      enableReinitialize: props.loadOnMount,
      isLoad: mailState.isLoad,
      autoSaveDate: mailState.autoSaveDate,
      initialValues: getInitialValues(mailState.data),
    };
  };

  const actions = {
    onSubmit: (values, isAutoSave) => {
      let result;
      if (typeof props.onSubmit === 'function') {
        result = props.onSubmit(values);
        if (!isAutoSave) {
          if (isPromise(result)) {
            result.then(() => {
              dispatch(pushNotification(values));
            });
          } else {
            dispatch(pushNotification(values));
          }
        }
      }
      return result;
    },
    autoSaveSuccess: (data) => dispatch(autoSaveSuccess(props.name, data)),
    onSubmitFail: (error, _, submitError) => {
      // eslint-disable-line no-shadow
      if (submitError.data && submitError.data.mailId) {
        dispatch(change(props.name, 'mailId', submitError.data.mailId));
      }
    },
    resetOpportunities: () => {
      dispatch(change(props.name, 'opportunities', []));
    },
    onCancel: () => {
      if (typeof props.onCancel === 'function') {
        props.onCancel();
      }
    },
    load: () => {
      switch (props.type) {
        case 'draft':
          dispatch(loadMail(props.name, props.loadDraft(props.loadArgs)));
          break;
        case 'reply':
          dispatch(loadMail(props.name, props.loadReply(props.loadArgs)));
          break;
        default:
          dispatch(loadMail(props.name, props.loadNew(props.loadArgs)));
          break;
      }
    },
  };

  return (nextState, nextOwnProps: Props): Props => {
    props = nextOwnProps;

    const nextPropsResult: Props = {
      ...nextOwnProps,
      ...getStateProps(nextState),
      ...actions,
    };

    if (!shallowEqual(resultProps, nextPropsResult)) resultProps = nextPropsResult;
    return resultProps;
  };
}

export default connectAdvanced(selectorFactory);
