import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { change } from 'redux-form';
import { Account } from 'types';
import { DocviewerFile, ImageFile } from 'components/FilesList/types';
import { get } from 'api/common';
import arrayReducerFactoryRaw from 'utils/reducer/arrayReducerFactoryRaw';
import { push } from 'modules/notificationsUI/actions';
import { NAME } from './constants';

type File = DocviewerFile | ImageFile;

interface LoadSuccessPayload {
  mailReply: {
    account: Account;
    body: string;
    froms: string[];
    isExternalRecipients: boolean;
    isHtml: boolean;
    isImportant: boolean;
    replyToMailId: number;
    subject: string;
    to: string;
  };
  success: boolean;
}

interface ActionMeta {
  name: string;
}

export interface Template {
  bodyHtml: string;
  bodyPlain: string;
  files: File[];
  name: string;
  orderPosition: number;
  templateId: number;
  type: string;
  typeName: string;
}

export interface Signature {
  bodyHtml: string;
  bodyPlain: string;
  isDefault: boolean;
  orderPosition: number;
  signatureId: number;
  templateId: number;
  type: string;
  typeName: string;
}

interface TemplateSuccessPayload {
  templates: Template[];
  signatures: Signature[];
}

type PassMeta = <P>() => (
  name: string,
  payload: P,
) => {
  payload: P;
  meta: {
    name: string;
  };
};
const passMeta: PassMeta = () => (name, payload) => ({
  payload,
  meta: {
    name,
  },
});

type State = {
  isFetch?: boolean;
  isFetchTemplates?: boolean;
};

const newMailSlice = createSlice({
  name: `${NAME}`,
  initialState: {
    isFetch: false,
    isFetchTemplates: false,
  } as State,
  reducers: {
    loadRequest: (state) => {
      return {
        ...state,
        isFetch: true,
      };
    },
    loadSuccess: {
      reducer: (state, action: PayloadAction<LoadSuccessPayload, string, ActionMeta>) => {
        const newState = {
          ...state,
          isLoad: true,
          data: action.payload.mailReply,
        };

        delete newState.isFetch;

        return newState;
      },
      prepare: passMeta<LoadSuccessPayload>(),
    },

    loadFail: (state) => {
      const newState = { ...state };
      delete newState.isFetch;
      return newState;
    },

    templatesRequest: (state) => {
      return {
        ...state,
        isFetchTemplates: true,
      };
    },

    templatesSuccess: {
      reducer: (state, action: PayloadAction<TemplateSuccessPayload, string, ActionMeta>) => {
        const newState = {
          ...state,
          isLoadTemplates: true,
          templates: action.payload,
        };

        delete newState.isFetchTemplates;

        return newState;
      },
      prepare: passMeta<TemplateSuccessPayload>(),
    },

    templatesFail: (state) => {
      const newState = { ...state };
      delete newState.isFetchTemplates;
      return newState;
    },

    autoSave: {
      reducer: (state, action: PayloadAction<string, string, ActionMeta>) => {
        return { ...state, autoSaveDate: action.payload };
      },
      prepare: passMeta<string>(),
    },

    destroy: {
      reducer: () => undefined,
      prepare: passMeta<undefined>(),
    },
  },
});

const {
  loadRequest,
  loadSuccess,
  loadFail,
  templatesRequest,
  templatesSuccess,
  templatesFail,
  autoSave,
} = newMailSlice.actions;

// eslint-disable-next-line
const loadMail = (name: string, promise: Promise<any>) => ({
  type: 'PROMISE',
  promise,
  actions: [loadRequest.type, loadSuccess.type, loadFail.type],
  meta: { name },
  decorate: {
    success: (action) => ({
      type: action.type,
      args: action.args,
      payload: action.data,
      meta: action.meta,
    }),
  },
});

const loadTemplates = (
  name: string,
  location: string | undefined,
  issueId: number | undefined,
  isInternalMail?: boolean,
) => ({
  type: 'PROMISE',
  promise: get({
    url: '/view/mail/template/list',
    global: false,
    data: {
      location,
      issueId,
      isInternalMail,
    },
  }).catch(() => ({
    signatures: [],
    templates: [],
  })),
  actions: [templatesRequest.type, templatesSuccess.type, templatesFail.type],
  meta: { name },
  decorate: {
    success: (action) => ({
      type: action.type,
      args: action.args,
      payload: action.data,
      meta: action.meta,
    }),
  },
});

// eslint-disable-next-line
const pushNotification = (args: any) => push({ view: 'SEND_MAIL', theme: 'success', args });

type AutoSaveSuccess = (
  name: string,
  data?: {
    mailId?: number;
  },
) => void;
const autoSaveSuccess: AutoSaveSuccess = (name, data) => (dispatch) => {
  if (data && data.mailId) {
    dispatch(change(name, 'mailId', data.mailId));
    dispatch(autoSave(name, new Date().toISOString()));
  }
};

const actions = {
  ...newMailSlice.actions,
  loadMail,
  loadTemplates,
  pushNotification,
  autoSaveSuccess,
};

const behaviors = Object.entries(newMailSlice.caseReducers).reduce((acc, [type, reducer]) => {
  acc[`${NAME}/${type}`] = reducer;
  return acc;
}, {});
const reducer = arrayReducerFactoryRaw([], behaviors);

export { reducer, actions };
