const WRAPPER_PLAIN = '\n\n';
const WRAPPER_HTML = '<div><br /></div><div><br /></div>';

export interface Signature {
  bodyHtml: string;
  bodyPlain: string;
  isDefault: true;
  signatureId: number;
}

export default class SignatureEditor {
  private signaturesMap = new Map<number, Signature>();

  public defaultSignatureId: number;

  public constructor(signatures: Signature[]) {
    this.setSignatures(signatures);
  }

  public setSignatures(signatures: Signature[]): void {
    this.signaturesMap.clear();

    signatures.forEach(value => {
      this.signaturesMap.set(value.signatureId, value);

      if (value.isDefault) {
        this.defaultSignatureId = value.signatureId;
      }
    });
  }

  public updateSignature(
    text: string,
    isHtml: boolean,
    signatureId: number,
    currentSignatureId?: number,
  ): string | null {
    if (isHtml) {
      return this.replaceSignatureHtml(text, signatureId);
    }

    return this.insertSignaturePlain(text, signatureId, currentSignatureId);
  }

  public insertSignaturePlain(
    text: string,
    signatureId: number,
    currentSignatureId?: number,
  ): string {
    const signature = this.getSignatureById(signatureId);

    if (!signature) {
      return text;
    }

    if (currentSignatureId) {
      const currentSignature = this.getSignatureById(currentSignatureId);

      if (currentSignature && text.indexOf(currentSignature.bodyPlain) !== -1) {
        return text.replace(currentSignature.bodyPlain, signature.bodyPlain);
      }
    }

    return WRAPPER_PLAIN + signature.bodyPlain + WRAPPER_PLAIN + text;
  }

  public insertSignatureHtml(text: string, signatureId: number): string {
    const signature = this.getSignatureById(signatureId);

    if (!signature) {
      return text;
    }

    const replaceResult = this.replaceSignatureHtml(text, signatureId);

    if (replaceResult) {
      return replaceResult;
    }

    return `${this.getSignatureHtmlTemplate(signatureId)}${text}`;
  }

  public getSignatureHtmlTemplate(signatureId: number): string | null {
    const signature = this.getSignatureById(signatureId);

    if (!signature) {
      return null;
    }

    return `${WRAPPER_HTML}<div class="spSignature">${signature.bodyHtml}</div>${WRAPPER_HTML}`;
  }

  public replaceSignatureHtml(text: string, signatureId: number): string | null {
    const signature = this.getSignatureById(signatureId);

    if (!signature) {
      return null;
    }

    let signatureReplaced = false;

    const div = document.createElement('div');
    div.innerHTML = text;

    div.querySelectorAll('.spSignature').forEach(signatureDivWrap => {
      if (signatureReplaced) {
        return;
      }

      const blockquote = signatureDivWrap.closest('blockquote');
      if (blockquote) {
        return;
      }

      // eslint-disable-next-line no-param-reassign
      signatureDivWrap.innerHTML = signature.bodyHtml;
      signatureReplaced = true;
    });

    if (signatureReplaced) {
      return div.innerHTML;
    }

    return null;
  }

  public getSignatureById(signatureId): Signature | undefined {
    return this.signaturesMap.get(signatureId);
  }
}
