import React from 'react';
import { connect } from 'react-redux';
import { PanelProps, Settings } from '@crm/components/dist/Panel';
import { MediaMatchContext, DeviceType } from 'components/MediaMatch';
import resizableRedux from 'modules/resizable/redux';
import { Panel } from './Panel';
import redux from '../../redux';
import {
  ConnectedState,
  ConnectedDispatch,
  ConnectedProps,
  IPanelContainerProps,
} from '../../types';

class PanelContainer extends React.Component<IPanelContainerProps> {
  public static contextType = MediaMatchContext;

  private previousContext: DeviceType;

  public componentDidMount(): void {
    this.previousContext = this.context;
    if (this.props.pin == null && this.props.panelId == null) {
      this.setSettings();
    }
  }

  public componentDidUpdate(): void {
    if (this.context !== this.previousContext) {
      if (this.context === DeviceType.Phone) {
        this.props.setPanelId(null);
      }

      this.previousContext = this.context;
    }
  }

  public componentWillUnmount(): void {
    const { panelId, pin, setPanelId } = this.props;

    if (!pin && panelId) {
      setPanelId(0);
    }
  }

  private setSettings() {
    const { defaultPin, defaultPanelId, setSettings, pin, panelId } = this.props;
    const payload: Settings = {};
    if (this.context === DeviceType.Phone) {
      payload.panelId = null;
    } else if (panelId == null && pin == null && (defaultPin || defaultPanelId)) {
      if (defaultPin) {
        payload.pin = defaultPin;
      }

      if (defaultPanelId) {
        payload.panelId = defaultPanelId;
      }
    }

    setSettings(payload);
  }

  public render(): React.ReactElement {
    const deviceType = this.context;
    let patchedProps = this.props as PanelProps;
    if (deviceType === DeviceType.Phone) {
      patchedProps = {
        ...this.props,
        pin: false,
        hasPinButton: false,
        canResize: false,
        width: 'max',
      } as PanelProps;
    }

    return <Panel {...patchedProps} />;
  }
}

export default connect<ConnectedState, ConnectedDispatch, ConnectedProps>(
  (state, props) => {
    return {
      ...redux.selectors.getStateByNameOrObject(state, props.name),
      size: props.size || resizableRedux.selectors.getSizeByName(state, props.name),
    };
  },
  (dispatch, props) => ({
    onPinClick: (name, pin) => dispatch(redux.actions.pin(name, pin)),
    setSettings: (payload) => dispatch(redux.actions.setSettings(props.name, payload)),
    setPanelId: (payload) => dispatch(redux.actions.setPanelId(props.name, payload)),
    onSizeChange: (size) => dispatch(resizableRedux.actions.setSize(props.name, size)),
  }),
)(PanelContainer);
