import React from 'react';
import PropTypes from 'prop-types';
import Duration from 'components/Duration';
import Icon from 'components/Icons/Icon';
import cx from 'classnames';
import css from './CallBar.modules.scss';

export default class CallBar extends React.Component {
  static propTypes = {
    phoneNumber: PropTypes.string,
    startCallDate: PropTypes.number,
    endCallDate: PropTypes.number,
    isAlreadySend: PropTypes.bool,
  };

  static defaultProps = {
    phoneNumber: undefined,
    startCallDate: undefined,
    endCallDate: undefined,
    isAlreadySend: false,
  };

  constructor(props) {
    super(props);
    const { startCallDate } = props;

    let duration = 0;
    if (startCallDate) {
      duration = Date.now() / 1000 - props.startCallDate;
    }

    if (duration < 0) {
      duration = 0;
    }

    this.state = {
      duration,
    };
  }

  componentDidMount() {
    this.interval = setInterval(this.timer, 1000);
  }

  componentWillReceiveProps(nextProps) {
    const { startCallDate, endCallDate } = nextProps;
    if (nextProps.endCallDate !== this.props.endCallDate) {
      clearInterval(this.interval);
      this.setState({ duration: endCallDate - startCallDate });
    }
  }

  componentWillUnmount() {
    clearInterval(this.interval);
  }

  timer = () => {
    this.setState(prevState => ({
      duration: prevState.duration + 1,
    }));
  };

  render() {
    const { phoneNumber, isAlreadySend, endCallDate } = this.props;
    const { duration } = this.state;

    if (isAlreadySend) {
      return (
        <div className={css.SendedCall}>
          <Icon icon="warning" />
          &nbsp; Карточка для этого звонка уже была сохранена
        </div>
      );
    }

    return (
      <div className={cx(css.CallBar, { [css.CallBar_end]: endCallDate })}>
        {phoneNumber && (
          <div className={css.CallBar__col}>
            <span className={css.CallBar__label}>Телефон</span>
            <span className={css.CallBar__value}>{phoneNumber}</span>
          </div>
        )}
        <div className={css.CallBar__col}>
          <span className={css.CallBar__label}>Продолжительность</span>
          <span className={css.CallBar__value}>
            <Duration duration={duration * 1000} />
          </span>
        </div>
      </div>
    );
  }
}
