import features from 'constants/features';
import React, { useCallback } from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import { reduxForm, Field, FieldArray, propTypes, getFormValues, change } from 'redux-form';
import withReduxFormProps from 'requestForms/components/withReduxFormProps';
import FormWrap from 'requestForms/components/FormWrap';
import FieldTreeSelect from 'requestForms/components/FieldTreeSelect';
import cx from 'classnames';
import Access from 'utils/Access';
import { actions as iframeActions } from 'components/Iframe';
import SuggestGrid from 'lego/redux-form/SuggestGrid';
import TextInput from 'lego/redux-form/TextInput';
import TextArea from 'lego/redux-form/TextArea';
import CheckBox from 'lego/redux-form/CheckBox';
import { TagsField as Tags, apiMail as apiTags } from 'modules/tags';
import Categorization from 'lego/redux-form/Categorization';
import PreviewValue from 'components/PreviewValue';
import DictInput from 'components/DictInput/DictInput';
import panelRedux from 'modules/panel/redux';
import { PANEL_NAME, TIPS_PANEL_ID } from '../Panel';
import css from './CallForm.modules.scss';

const canItemClick = (item) => item.isLeaf;

const DEFAULT_REQUIRE_MARKUP = 1;

// eslint-disable-next-line
const ReduxFormPreviewValue = ({ input, meta, ...other }) => (
  <PreviewValue {...other} value={input.value} />
);

const ReduxFormDictInput = withReduxFormProps()((props) => {
  const { onChange, ...other } = props;

  const handleChange = useCallback((id, option) => onChange(option), [onChange]);

  return <DictInput {...other} onChange={handleChange} />;
});

const ReduxFormDictInputPreview = ({ input, meta, ...other }) => (
  <PreviewValue {...other} value={input.value && input.value.name} />
);

const CallForm = (props) => {
  const { handleSubmit, access, autoFocus, account, formValues = {}, resetOpportunities } = props;

  React.useEffect(() => {
    if (resetOpportunities) resetOpportunities();
  }, [account?.id]);

  const accessMap = {
    category: Access.isEdit(access.category),
    categories: Access.isEdit(access.categories),
    rejectUpsale: Access.isEdit(access.isRejectUpsale),
    rejectMail: Access.isEdit(access.isRejectMail),

    inn: Access.isRead(access.inn),
    contractNum: Access.isRead(access.contractNum),
    businessUnit: Access.isEdit(access.businessUnit),
    service: Access.isEdit(access.service),
  };

  return (
    <FormWrap onSubmit={handleSubmit}>
      <div className={cx(css.row, css.row_manyFields, css.row_stretch, css.row_margin_bottom)}>
        <div className={css.field}>
          <span className={css.field__label}>Имя</span>
          <Field
            className={css.field__input}
            name="callerName"
            placeholder="Имя"
            component={TextInput}
          />
        </div>
        {accessMap.category && (
          <div className={cx(css.field, css.field_category)}>
            <span className={cx(css.field__label, css.field__label_required)}>Категория</span>
            <Field
              className={css.field__input}
              component={FieldTreeSelect}
              name="category"
              mode="radio"
              endpoint="/view/support/category/list?workplace=phone"
              props={{ name: 'category' }}
              canItemClick={canItemClick}
              defaultValueText={'\u2014'}
              displayKeySlug="fullPath"
            />
          </div>
        )}
      </div>
      {(accessMap.inn || accessMap.contractNum || accessMap.businessUnit || accessMap.service) && (
        <div className={cx(css.row, css.row_manyFields, css.row_margin_bottom)}>
          {accessMap.inn && (
            <div className={css.field}>
              <span className={css.field__label}>ИНН</span>
              <Field name="inn" className={css.field__input} component={ReduxFormPreviewValue} />
            </div>
          )}
          {accessMap.contractNum && (
            <div className={css.field}>
              <span className={css.field__label}>№ договора</span>
              <Field
                name="contractNum"
                className={css.field__input}
                component={ReduxFormPreviewValue}
              />
            </div>
          )}
          {accessMap.businessUnit && (
            <div className={css.field}>
              <span className={css.field__label}>Бизнес юнит</span>
              <Field
                name="businessUnit"
                className={cx(css.field__input, css.field__dictInputValue)}
                component={ReduxFormDictInputPreview}
              />
              <Field
                name="businessUnit"
                className={css.field__input}
                component={ReduxFormDictInput}
                type={DictInput.Types.BusinessUnit}
              />
            </div>
          )}
          {accessMap.service && (
            <div className={css.field}>
              <span className={css.field__label}>Сервис</span>
              <Field
                name="service"
                className={cx(css.field__input, css.field__dictInputValue)}
                component={ReduxFormDictInputPreview}
              />
              <Field
                name="service"
                className={css.field__input}
                component={ReduxFormDictInput}
                type={DictInput.Types.Service}
              />
            </div>
          )}
        </div>
      )}

      {!formValues.noLinkWithOpportunity && account?.id && (
        <div className={cx(css.row, css.row_margin_bottom)}>
          <div className={css.field}>
            <span className={css.field__label}>Сделки</span>
            <Field
              name="opportunities"
              component={SuggestGrid}
              provider={`/v0/blocks/opportunity/suggest?accountId=${account.id}`}
              account={account}
              title="Привяжите сделки"
            />
          </div>
        </div>
      )}
      <div className={cx(css.row, css.row_margin_bottom_l)}>
        <div className={css.field}>
          <Field name="noLinkWithOpportunity" component={CheckBox} onChange={resetOpportunities}>
            Звонок не относится к сделке
          </Field>
        </div>
      </div>

      <Field name="comment" component={TextArea} placeholder="Комментарий" autoFocus={autoFocus} />
      {(accessMap.rejectUpsale || accessMap.rejectMail) && (
        <div className={cx(css.row, css.row_manyFields)}>
          {accessMap.rejectUpsale && (
            <div className={css.field}>
              <Field name="isRejectUpsale" component={CheckBox}>
                Ничего не предлагать
              </Field>
            </div>
          )}
          {accessMap.rejectMail && (
            <div className={css.field}>
              <Field name="isRejectMail" component={CheckBox}>
                Отказ от письма
              </Field>
            </div>
          )}
        </div>
      )}
      <FieldArray name="tags" component={Tags} getTags={apiTags.get} />
      {accessMap.categories && (
        <Field
          name="categories"
          className={css.categorization}
          component={Categorization}
          isRequireMarkup={DEFAULT_REQUIRE_MARKUP}
          workplace="SpCall"
          isOpen
          onTipClick={props.openTip}
        />
      )}
    </FormWrap>
  );
};

CallForm.propTypes = {
  ...propTypes,
  access: PropTypes.instanceOf(Object),
  autoFocus: PropTypes.bool,
};

CallForm.defaultProps = {
  access: {},
  autoFocus: false,
};

export default connect(
  (state, props) => ({
    formValues: getFormValues(props.form)(state),
  }),
  (dispatch, props) => ({
    openTip: (url) => {
      if (features.openTipInNewTab) {
        window.open(url);
      } else {
        dispatch(iframeActions.setSrc(url));
        dispatch(panelRedux.actions.setPanelId(PANEL_NAME, TIPS_PANEL_ID, false));
      }
    },
    resetOpportunities: () => {
      dispatch(change(props.form, 'opportunities', []));
    },
  }),
)(reduxForm()(CallForm));
