import React from 'react';
import PropTypes from 'prop-types';
import PlainText from 'components/PlainText';
import Duration from 'components/Duration';
import LegoTags from 'lego/components/Tags';
import cx from 'classnames';
import Access from 'utils/Access';
import { CategoriesPreview } from 'modules/categorization';
import { Bubble } from 'components/Bubble';
import PreviewValue from 'components/PreviewValue';
import { TagsWithApi as Tags } from 'modules/tags';
import css from './CallForm.modules.scss';
import chipsCss from './chips.scss';

const CallFormPreview = (props) => {
  const {
    values: {
      id,
      callerName,
      category,
      categories,
      comment,
      hasMail,
      isRejectUpsale,
      isRejectMail,
      factors,
      tags,
      recordLink,
      duration,
      phoneNumber,
      inn,
      contractNum,
      businessUnit,
      service,
      opportunities,
      noLinkWithOpportunity,
    },
    access,
  } = props;

  const accessMap = {
    inn: Access.isRead(access.inn),
    contractNum: Access.isRead(access.contractNum),
    category: Access.isRead(access.category),
    categories: Access.isRead(access.categories),
    tags: {
      read: Access.isRead(access.tags),
      edit: Access.isEdit(access.tags),
    },
    businessUnit: Access.isRead(access.businessUnit),
    service: Access.isRead(access.service),
  };

  return (
    <div>
      {callerName && (
        <div className={css.field}>
          <span className={css.field__label}>Имя</span>
          <span className={css.field__input}>{callerName}</span>
        </div>
      )}
      {(accessMap.inn || accessMap.contractNum || accessMap.service || accessMap.businessUnit) && (
        <div className={cx(css.row, css.row_manyFields)}>
          {accessMap.inn && (
            <div className={css.field}>
              <span className={css.field__label}>ИНН</span>
              <PreviewValue className={css.field__input} value={inn} />
            </div>
          )}
          {accessMap.contractNum && (
            <div className={css.field}>
              <span className={css.field__label}>№ договора</span>
              <PreviewValue className={css.field__input} value={contractNum} />
            </div>
          )}
          {accessMap.businessUnit && (
            <div className={css.field}>
              <span className={css.field__label}>Бизнес юнит</span>
              <PreviewValue
                className={cx(css.field__input, css.field__input_limited)}
                value={businessUnit && businessUnit.name}
              />
            </div>
          )}
          {accessMap.service && (
            <div className={css.field}>
              <span className={css.field__label}>Сервис</span>
              <PreviewValue
                className={cx(css.field__input, css.field__input_limited)}
                value={service && service.name}
              />
            </div>
          )}
        </div>
      )}
      {duration && (
        <div className={css.field}>
          <span className={css.field__label}>Длительность</span>
          <span className={css.field__input}>
            <Duration duration={duration * 1000} />
          </span>
        </div>
      )}

      <div className={cx(css.row, css.row_margin_bottom)}>
        <div className={css.field}>
          {noLinkWithOpportunity ? (
            <span className={css.field__label}>Звонок не относится к сделке</span>
          ) : (
            <>
              <span className={css.field__label}>Сделки</span>
              {opportunities?.length ? (
                <LegoTags items={opportunities} component={Bubble} />
              ) : (
                <span>&mdash;</span>
              )}
            </>
          )}
        </div>
      </div>

      {phoneNumber && (
        <div className={css.field}>
          <span className={css.field__label}>Телефон</span>
          <span className={css.field__input}>{phoneNumber}</span>
        </div>
      )}
      {accessMap.category && category && (
        <div className={cx(css.field, css.field_category, css.field_preview)}>
          <span className={css.field__label}>Категория</span>
          <span className={css.field__input}>{category.name}</span>
        </div>
      )}
      {accessMap.categories && (
        <div className={cx(css.field, css.field_category, css.field_preview)}>
          <span className={css.field__label}>Категории</span>
          <div className={css.field__input}>
            <CategoriesPreview categories={categories} />
          </div>
        </div>
      )}
      {comment && (
        <div className={cx(css.field, css.field_comment)}>
          <span className={css.field__label}>Комментарий</span>
          <PlainText text={comment} link />
        </div>
      )}
      <div className={cx(css.row, css.row_manyFields)}>
        {hasMail && <div className={chipsCss.default}>Письмо</div>}
        {isRejectMail && <div className={chipsCss.warning}>Отказ от письма</div>}
        {isRejectUpsale && <div className={chipsCss.warning}>Ничего не предлагать</div>}
        {Array.isArray(factors) &&
          factors.length > 0 &&
          factors.map((item) => (
            <div key={item} className={chipsCss.factor}>
              {item}
            </div>
          ))}
      </div>
      {recordLink && (
        <div>
          <a href={recordLink} target="_blank" rel="noopener noreferrer">
            Запись разговора
          </a>
        </div>
      )}
      {accessMap.tags.read && (
        <Tags
          key={`call${id}`}
          tags={tags}
          objectType="call"
          objectId={id}
          canEdit={accessMap.tags.edit}
        />
      )}
    </div>
  );
};

CallFormPreview.propTypes = {
  values: PropTypes.instanceOf(Object),
  access: PropTypes.instanceOf(Object),
};

CallFormPreview.defaultProps = {
  values: {},
  access: {},
};

export default CallFormPreview;
