import React from 'react';
import PropTypes from 'prop-types';
import get from 'lodash/get';
import cx from 'classnames';
import { FormattedDate } from 'react-intl';
import PlainText from 'components/PlainText';
import AccountLink from 'pages/accounts/components/AccountLink';
import { Link } from 'react-router-dom';
import Duration from 'components/Duration';
import { CallTo } from 'modules/phone';
import CallIcon from '../CallIcon';
import css from './TableCall.modules.scss';

const textRender = value => <PlainText text={value} link />;

const renderAccount = value => <AccountLink account={value} />;

const renderPhone = (value, { phoneNumberE164, account }) => (
  <span>
    {value}
    {phoneNumberE164 ? ' ' : undefined}
    {phoneNumberE164 && (
      <CallTo phone={phoneNumberE164} cardData={{ accountId: get(account, 'id') }}>
        позвонить
      </CallTo>
    )}
  </span>
);

const renderDate = (value, column) => (
  <span className={css[`date_${column.type}`]}>
    <CallIcon type={column.type} />
    &nbsp;
    <FormattedDate
      value={new Date(value)}
      year="numeric"
      month="numeric"
      day="numeric"
      hour="numeric"
      minute="numeric"
    />
    &nbsp;
    <Duration duration={column.duration * 1000} />
  </span>
);

const CUSTOM_RENDER = {
  account: renderAccount,
  date: renderDate,
  result: textRender,
  comment: textRender,
  phoneNumber: renderPhone,
};

const TableCall = props => {
  const { item = {}, columns, className, cellClassName, columnsStyle, index } = props;

  const { callId } = item;

  let Component = 'div';
  let rootProps;
  if (callId) {
    Component = Link;
    rootProps = { to: `/phone/${callId}`, target: '_blank' };
  }

  return (
    <Component className={cx(className, css.TableCall)} {...rootProps}>
      <div
        className={cx(cellClassName, css[`TableCall__index_${item.type}`])}
        style={columnsStyle && columnsStyle.index}
      >
        {index}
      </div>
      {columns.map(column => {
        let value = item[column.name];
        const customRender = CUSTOM_RENDER[column.name];
        if (customRender) {
          value = customRender(value, item);
        }

        return (
          <div
            key={column.name}
            className={cellClassName}
            style={columnsStyle && columnsStyle[column.name]}
          >
            {value}
          </div>
        );
      })}
    </Component>
  );
};

TableCall.propTypes = {
  item: PropTypes.objectOf(PropTypes.any).isRequired,
  columns: PropTypes.objectOf(PropTypes.any).isRequired,
  columnsStyle: PropTypes.objectOf(PropTypes.any),
  index: PropTypes.number.isRequired,
  cellClassName: PropTypes.string,
  className: PropTypes.string,
};

TableCall.defaultProps = {
  cellClassName: undefined,
  columnsStyle: undefined,
  className: undefined,
};

export default TableCall;
