import * as React from 'react';
import { connect } from 'react-redux';
import { Account } from 'types';
import { PanelId, PanelScheme } from '@crm/components/dist/Panel';
import Panel from 'modules/panel';
import Iframe, { selectors as iframeSelectors } from 'components/Iframe';
import { PanelFiles, PanelEntity } from 'modules/clientHistory';
import { getNewCall } from '../../selectors';
import css from './Panel.module.css';

interface OwnProps {
  account: Account;
}

interface ConnectedState {
  panelId: PanelId;
  tipUrl: string;
  account: Account;
}

interface ConnectedDispatch {
  onPanelSet: (panelId: PanelId) => void;
}

type Props = OwnProps & ConnectedState & ConnectedDispatch;

export const PANEL_NAME = 'newCall/categorization';
export const TIPS_PANEL_ID = 'categorizationTips';

const NewCallPanel: React.FC<Props> = (props) => {
  const { account, tipUrl } = props;
  const panels: PanelScheme[] = [];

  if (tipUrl) {
    panels.push({
      id: TIPS_PANEL_ID,
      title: 'Подсказки',
      content: (
        <div className={css.panelWrap}>
          <Iframe title="Подсказки" />
        </div>
      ),
    });
  }

  if (account) {
    const accountId = account.id;

    panels.push({
      id: 'activity',
      title: 'История аккаунта',
      content: (
        <div className={css.panelWrap}>
          <PanelEntity
            key={accountId}
            param={{ clientId: accountId }}
            name="activity"
            loadOnMount
          />
        </div>
      ),
      keepMount: true,
      padding: false,
    });

    panels.push({
      id: 'files',
      title: 'Файлы аккаунта',
      content: (
        <div className={css.panelWrap}>
          <PanelFiles
            key={accountId}
            param={{ clientId: accountId }}
            name="files"
            loadOnMount
            showTitle={false}
          />
        </div>
      ),
      keepMount: true,
      padding: false,
    });
  }

  return panels.length ? (
    <Panel
      name={PANEL_NAME}
      defaultSize={{
        width: 300,
      }}
      minWidth={300}
      maxWidth={490}
      panels={panels}
    />
  ) : null;
};

export default connect(
  (state, ownProps: OwnProps): ConnectedState => {
    const { panelId }: { panelId: PanelId } = getNewCall(state);
    const { account } = ownProps;
    const iframeSrc: string = iframeSelectors.getState(state);

    return {
      panelId,
      account,
      tipUrl: iframeSrc,
    };
  },
)(NewCallPanel);
