import { jsonApiCall } from 'api/common';
import { getFormValues, submit } from 'redux-form';
import { PinFormModal } from 'modules/pinForm';
import favicon from 'utils/favicon';
import {
  CALL_FORM_NAME,
  LOCAL_STORAGE_SEND_CALL_ID_KEY,
  LOCAL_STORAGE_PHONE_OUT_SESSION_ID,
  LOCAL_STORAGE_PHONE_OUT_CARD_DATA,
  CALL_MAIL_NAME,
} from 'modules/phone/constants';
import filterValuesToBackend from 'modules/phone/utils/filterValuesToBackend';
import getCallGuid from 'modules/phone/utils/getCallGuid';
import { getAccountId, getActiveCall, getNewCall, getNewCallId } from 'modules/phone/selectors';
import clearDataCrmProps from 'utils/clearDataCrmProps';
import { push } from 'modules/notificationsUI/actions';
import parseFromBackend from 'modules/categorization/utils/parseFromBackend';
import * as Upsale from 'modules/upsale';

import {
  DESTROY,
  NEW_CALL_RECEIVE,
  NEW_SAVE_RECEIVE,
  SET_ACCOUNT,
  SET_CALL_DATA,
  SET_CALL_INFO,
  SET_CALL_MAIL,
  SET_ALREADY_SEND,
  SET_SEND,
  SET_SAVE,
  SET_ADDRESSEE_DETAILS,
  SET_TIP,
} from '../actionTypes/newCall';

export const getAddresseeDetails = (account) => (dispatch) =>
  jsonApiCall({
    url: '/view/telephony/addressee',
    data: account,
  }).then((data) => {
    dispatch({ type: SET_ADDRESSEE_DETAILS, payload: data });
    return data;
  });

export const getTip = (account, callId) => (dispatch) =>
  jsonApiCall({
    url: '/view/telephony/tip',
    data: { ...account, callId },
  }).then((data) => {
    dispatch({ type: SET_TIP, payload: data });
    return data;
  });

export const setAccount = (account) => (dispatch, getState) => {
  dispatch({ type: SET_ACCOUNT, payload: account });
  dispatch(getAddresseeDetails(account));
  dispatch(getTip(account, getNewCallId(getState())));
  // eslint-disable-next-line no-use-before-define
  dispatch(showPinModalIfNeeded(account && account.id));
};

export const submitPinForm = (values) => async (dispatch, getState) => {
  const call = getNewCall(getState());
  const callData = call && call.callData;
  const { pin } = values;

  const { account } = await jsonApiCall({
    url: '/actions/telephony/checkPin',
    data: {
      pin,
      callData,
    },
  });

  dispatch(setAccount(account || {}));
};

export const showPinForm = () => (dispatch) => {
  PinFormModal.open({
    onSubmit: (values) => dispatch(submitPinForm(values)),
  });
};

export function showPinModalIfNeeded(accountId) {
  return async (dispatch, getState) => {
    const newCall = getNewCall(getState());
    let callData = {};
    if (newCall) {
      callData = newCall.callData || {};
    }
    const { autoShowPin } = await jsonApiCall({
      url: '/view/telephony/showPin',
      data: {
        accountId,
        callData,
      },
    });

    if (autoShowPin) {
      dispatch(showPinForm());
    }

    return autoShowPin;
  };
}

export const setCallData = (payload) => ({ type: SET_CALL_DATA, payload });
export const setCallInfo = (payload) => {
  let patchedPayload = payload;
  if (payload && payload.categories) {
    patchedPayload = { ...payload, categories: parseFromBackend(payload.categories) };
  }

  return { type: SET_CALL_INFO, payload: patchedPayload };
};
export const setCallMail = (payload) => ({ type: SET_CALL_MAIL, payload });

const checkIfNewCallAlreadySend = (dispatch, currentCallId) => (e) => {
  if (e.key === LOCAL_STORAGE_SEND_CALL_ID_KEY) {
    const { newValue: lastSendCallId } = e;
    if (currentCallId === lastSendCallId) {
      dispatch({ type: SET_ALREADY_SEND });
    }
  }
};

let saveLinkFunc;

export const load = (callId) => (dispatch, getState) => {
  const activeCall = getActiveCall(getState());
  if (activeCall) {
    favicon.set(favicon.TYPES.CALL_ACTIVE);
  } else {
    favicon.set(favicon.TYPES.CALL_EMPTY);
  }
  dispatch(setCallData(activeCall));

  const callGuid = getCallGuid(activeCall);
  if (callGuid) {
    saveLinkFunc = checkIfNewCallAlreadySend(dispatch, callGuid);
    window.addEventListener('storage', saveLinkFunc);
  }

  let loadData;
  if (callId) {
    loadData = { callId };
  } else if (activeCall) {
    loadData = activeCall;
    if (localStorage.getItem(LOCAL_STORAGE_PHONE_OUT_SESSION_ID) === activeCall.session_id) {
      try {
        const cardData = JSON.parse(localStorage.getItem(LOCAL_STORAGE_PHONE_OUT_CARD_DATA));
        loadData = { ...loadData, ...cardData };
      } catch (e) {
        /* empty */
      }
    }
  }

  return jsonApiCall({
    url: '/view/telephony/card',
    data: loadData,
  }).then((data = {}) => {
    let payload;
    try {
      payload = {
        ...data,
        mail: {
          ...data.mail,
          body: data.mail.body.replace(/{{factors}}/, '<div data-crm-macro="factors">\u200B</div>'),
        },
      };
    } catch (e) {
      payload = data;
    }

    dispatch(showPinModalIfNeeded(data.callInfo && data.callInfo.accountId));
    dispatch({ type: NEW_CALL_RECEIVE, payload });
  });
};

// hack for forward call
let isDraft = false;

export const callEnd = (data) => (dispatch) => {
  dispatch(setCallData(data));
  isDraft = true;
  dispatch(submit(CALL_MAIL_NAME));
  favicon.set(favicon.TYPES.CALL_END);
};

export const destroy = () => (dispatch) => {
  isDraft = false;
  window.removeEventListener('storage', saveLinkFunc);
  saveLinkFunc = undefined;
  favicon.set(favicon.TYPES.DEFAULT);
  PinFormModal.destroy();
  dispatch({ type: DESTROY });
};

export const send = (values) => (dispatch, getState) => {
  const state = getState();
  const formValues = filterValuesToBackend(getFormValues(CALL_FORM_NAME)(state));
  const accountId = getAccountId(state);
  const account = accountId ? { id: accountId } : undefined;
  const newCall = getNewCall(state) || {};

  let mailData;
  try {
    const mailId = newCall.mail.id;
    if (mailId) {
      mailData = { ...values, mailId };
    } else {
      mailData = values;
    }
  } catch (e) {
    mailData = values;
  }

  // надо body удалить из form.values.body и брать его только при отправке
  try {
    mailData.body = window.CKEDITOR.instances.body.getData();
  } catch (e) {
    /* empty */
  }

  try {
    mailData.body = clearDataCrmProps(mailData.body);
  } catch (e) {
    /* empty */
  }

  const offeredFactors = Upsale.selectors.getUpsaleFactorsSelected(state) || [];
  const factorsForOffer = Upsale.selectors.getUpsaleFactorsForOffer(state) || [];
  const upsaleDraftId = Upsale.selectors.getUpsaleDraftId(state);
  const selectedFactors = Upsale.SelectedFactors.frontendToBackend(
    Upsale.selectors.getSelectedFactors(state),
  );

  const selfIsDraft = isDraft;
  isDraft = false;

  const sendingData = {
    callData: newCall.callData,
    callInfo: { ...newCall.callInfo, ...formValues },
    account,
    mail: mailData,
    offeredFactors,
    selectedFactors,
    factorsForOffer,
    upsaleDraftId,
    isDraft: selfIsDraft,
  };

  return jsonApiCall({
    url: '/actions/telephony/card/save',
    data: sendingData,
  }).then((data) => {
    if (data.success) {
      const { callInfo, mail } = data;

      if (callInfo) {
        dispatch(setCallInfo(callInfo));
      }

      if (mail) {
        dispatch(setCallMail(mail));
      }

      if (!selfIsDraft) {
        if (sendingData.mail && sendingData.mail.send) {
          dispatch(push({ theme: 'success', text: 'Письмо отправлено' }));
          dispatch({ type: SET_SEND });
        } else {
          dispatch(push({ theme: 'success', text: 'Карточка сохранена' }));
        }
        dispatch({ type: SET_SAVE });

        localStorage.setItem(LOCAL_STORAGE_SEND_CALL_ID_KEY, getCallGuid(newCall.callData));
        dispatch({ type: NEW_SAVE_RECEIVE });
      }
    } else {
      const { callInfo, mail, message } = data;

      if (callInfo) {
        dispatch(setCallInfo(callInfo));
      }
      if (mail) {
        dispatch(setCallMail(mail));
      }

      dispatch(push({ theme: 'error', text: message }));
    }

    return data;
  });
};
