import get from 'lodash/get';
import { SEND, CREATE } from 'middlewares/websocket';
import {
  BACKEND_CMD,
  NAME,
  LOCAL_STORAGE_PHONE_OUT_CARD_DATA,
  LOCAL_STORAGE_PHONE_OUT_SESSION_ID,
} from 'modules/phone/constants';
import { getNewCall } from 'modules/phone/selectors';
import getCallGuid from 'modules/phone/utils/getCallGuid';
import { LOGGER_ACTION_TYPE } from 'middlewares/logger';
import { callEnd } from './newCall';
import frontendStatus from '../frontendStatus';

export const change = (status) => ({
  type: SEND,
  name: NAME,
  data: {
    cmd: BACKEND_CMD.UPDATE_AGENT_STATE,
    data: {
      state: status,
    },
  },
});

let outcallCardData = null;
export const call = (phone, cardData) => (dispatch) => {
  outcallCardData = cardData;
  dispatch(frontendStatus.actions.outboundCall());
  dispatch({
    type: SEND,
    name: NAME,
    data: {
      cmd: BACKEND_CMD.CREATE_OUTBOUND_DIRECT_SESSION,
      data: {
        call_to: phone,
      },
    },
  });
};

export const getInterfaceSettings = () => ({
  type: SEND,
  name: NAME,
  data: {
    cmd: BACKEND_CMD.GET_DEFAULT_INTERFACE_NAME,
    data: {},
  },
});

export const setInterfaceSettings = (value) => ({
  type: SEND,
  name: NAME,
  data: {
    cmd: BACKEND_CMD.SET_DEFAULT_INTERFACE_NAME,
    data: {
      interface: value,
    },
  },
});

const behaviors = {
  [BACKEND_CMD.CALL_END](store) {
    return (data) => {
      const state = store.getState();
      const newCall = getNewCall(state);
      if (newCall) {
        const callEndId = getCallGuid(data);
        const newCallId = getCallGuid(newCall.callData);
        if (newCallId && newCallId === callEndId) {
          store.dispatch(callEnd(data));
        }
      }
    };
  },
  [BACKEND_CMD.CREATE_OUTBOUND_DIRECT_SESSION]() {
    return (_, data) => {
      if (
        outcallCardData &&
        data.result === 'ok' &&
        data.result_details &&
        data.result_details.session_id
      ) {
        try {
          localStorage.setItem(LOCAL_STORAGE_PHONE_OUT_SESSION_ID, data.result_details.session_id);
          localStorage.setItem(LOCAL_STORAGE_PHONE_OUT_CARD_DATA, JSON.stringify(outcallCardData));
        } catch (e) {
          /* empty */
        }
        outcallCardData = null;
      }
    };
  },
};

export const message = (store) => (messageData = {}) => {
  try {
    // eslint-disable-next-line camelcase
    const { result, result_details } = messageData;
    // eslint-disable-next-line camelcase
    if (result === 'error' && result_details) {
      store.dispatch({
        type: LOGGER_ACTION_TYPE,
        isLogOnBackend: false,
        errorObj: new Error(result_details),
        alertText: result_details,
        alertType: 'ERROR',
      });
    }
  } catch (e) {
    /* empty */
  }

  const behavior = behaviors[messageData.cmd];
  if (behavior) {
    behavior(store)(messageData.data, messageData);
  }
};

export const initWebsocket = () => (dispatch, getState) => {
  const state = getState();
  const login = get(state, 'root.info.user.login');
  const testMode = get(state, 'root.info.testMode');

  let websocketUrl;
  if (testMode) {
    websocketUrl = `wss://octopus-stats-test.paysys.yandex-team.ru:8804/websocket?staff_login=${login}`;
  } else {
    websocketUrl = `wss://octopus-stats.paysys.yandex-team.ru:8804/websocket?staff_login=${login}`;
  }

  dispatch({
    type: CREATE,
    name: NAME,
    url: websocketUrl,
    message,
  });
};
