import React from 'react';
import get from 'lodash/get';
import AttachFiles from 'lego/components/AttachFiles';
import Radiobox, { Radio } from '@crm/components/dist/lego2/Radiobox';
import Button from '@crm/components/dist/lego2/Button';
import cx from 'classnames';
import Link from '@crm/components/dist/lego2/react-router/Link';
import { Title } from '@crm/components/dist/Title';
import Label from 'lego/components/Label';
import YandexForm from 'components/YandexForm';
import { ModalForm as ClientModal } from 'modules/clientSelectForm';
import CompleteOverlay from 'components/CompleteOverlay';
import TreeViewModule from 'dialogs/TreeView';
import { PRIORITY } from './Form.config';
import css from './Form.module.css';
import {
  ReduxState,
  IssueCreateFormRequest,
  IssueCreateFormResponse,
  PostFormRequest,
  TreeFormItem,
} from '../../types';
import getIssueLink from '../../utils/getIssueLink';

const TreeView = TreeViewModule.component;

interface FormProps {
  data: IssueCreateFormRequest;
  dialog: ReduxState;
  reset: () => void;
  load: (data: IssueCreateFormRequest) => Promise<IssueCreateFormResponse>;
  onUpdateForm: (data: PostFormRequest) => void;
  onSubmit: (data: PostFormRequest) => void;
  onFilesAdd: (file: unknown) => void;
  onFileRemove: (data: unknown) => void;
  onPriorityChange: (data: unknown) => void;
  onSelectCategoryClick: (data: unknown) => void;
  onChangeCategory: (itemId: number, item: TreeFormItem) => void;
  showSuccessMessage: boolean;
  setClient: (...data: unknown[]) => void;
  title?: string;
}

// todo: refactor big class
export default class Form extends React.Component<FormProps> {
  public static defaultProps = {
    showSuccessMessage: true,
    title: 'Создать новый тикет',
  };

  public componentDidUpdate(prevProps) {
    const currentCatId = get(this.props, 'dialog.data.category.id');
    const catId = get(prevProps, 'dialog.data.category.id');
    if (currentCatId && currentCatId !== catId) {
      this.props.onUpdateForm(this.getData());
    }
  }

  public componentWillUnmount() {
    this.props.reset();
  }

  private getData = () => ({
    data: {
      categoryId: get(this.props, 'dialog.data.category.id'),
      accountId: get(this.props, 'dialog.data.account.info.id'),
      formId: get(this.props, 'dialog.data.form.id'),
      containerId: get(this.props, 'dialog.data.containerId'),
      priorityId: get(this.props, 'dialog.priority'),
      issueId: get(this.props, 'dialog.data.issueId'),
      parentId: this.props.data.data.issueId,
      email: this.props.data.data.email,
      byEObject: this.props.data.data.byEObject,
      byChat: this.props.data.data.byChat,
    },
    context: this.props.data.context,
  });

  private handleChangeAccount = (...args) => {
    this.props.setClient(...args);
    this.props.onUpdateForm(this.getData());
  };

  private handleSubmit = () => {
    this.props.onSubmit(this.getData());
  };

  private _handleFilesAdd = (files) => {
    if (this.props.onFilesAdd) {
      this.props.onFilesAdd({
        files,
        objectName: 'request',
        containerId: get(this.props, 'dialog.data.containerId'),
      });
    }
  };

  private _handleFileRemove = (fileId) => {
    if (this.props.onFileRemove) {
      this.props.onFileRemove({
        fileId,
        objectName: 'request',
        containerId: get(this.props, 'dialog.data.containerId'),
      });
    }
  };

  private _renderForm() {
    if (
      !this.props.dialog.data ||
      !this.props.dialog.data.form ||
      !this.props.dialog.data.form.extFormId
    ) {
      return null;
    }

    const { isFormHide } = this.props.dialog;

    const formId = this.props.dialog.data.form.extFormId;
    const { initializeFormFields } = this.props.dialog.data;
    const { issueId } = this.props.dialog.data;

    return (
      <YandexForm
        formId={formId}
        name="request_form"
        onSuccess={this.handleSubmit}
        queryObject={{ ...initializeFormFields, requestId: `i${issueId}` }}
        style={isFormHide ? { display: 'none' } : undefined}
        className={css.b__form}
      />
    );
  }

  private onFilesUpdate = () => {};

  public render() {
    const { dialog, onChangeCategory, title } = this.props;
    const { error, data } = dialog;

    const { category } = data;

    const categoryRender = () => {
      const inputCss =
        dialog.data.category && dialog.data.category.name
          ? css.selectInput
          : css.selectInputPlaceholder;
      const categoryText = (category && category.name) || 'не выбрана';

      const props: {
        className?: string;
        onClick?: () => void;
      } = {};
      if (!error) {
        props.className = inputCss;
        props.onClick = () => this.props.onSelectCategoryClick(category && category.id);
      }

      return (
        <div className={css.group}>
          <Label hasLine={false} required className={css.label}>
            Категория тикета
          </Label>
          <span className={css.field}>
            <span {...props}>{categoryText}</span>
          </span>
        </div>
      );
    };

    const clientRender = () => {
      let account;
      if (dialog.data.account) {
        account = `${dialog.data.account.info.name || '\u2014'} (${dialog.data.account.info.login ||
          '\u2014'})`;
      } else {
        account = 'не выбран';
      }

      const inputCss = dialog.data.account ? css.selectInput : css.selectInputPlaceholder;

      return (
        <div className={css.group}>
          <Label hasLine={false} required className={css.label}>
            Клиент
          </Label>
          <span className={css.field}>
            {error ? (
              <span>{account}</span>
            ) : (
              <ClientModal name="NEW_REQUEST_CLIENT_MODAL" onSubmit={this.handleChangeAccount}>
                {(modal, open) => (
                  <span className={inputCss} onClick={open}>
                    {account}
                    {modal}
                  </span>
                )}
              </ClientModal>
            )}
          </span>
        </div>
      );
    };

    const renderPriority = () => (
      <div className={cx(css.group)}>
        <Label hasLine={false} required className={css.label}>
          {PRIORITY.caption}
        </Label>
        <span className={cx(css.field, css.field_input)}>
          <Radiobox
            size="m"
            onChange={this.props.onPriorityChange}
            value={String(dialog.priority)}
            isEventValue={false}
            name="priority"
            disabled={error}
          >
            {PRIORITY.list.map((id) => (
              <Radio key={id} value={String(id)}>
                {PRIORITY.map[id]}
              </Radio>
            ))}
          </Radiobox>
        </span>
      </div>
    );

    let message;
    if (
      this.props.showSuccessMessage &&
      this.props.dialog.status &&
      this.props.dialog.receiveData
    ) {
      const issueId = this.props.dialog.receiveData.id;

      message = (
        <div className={css.message}>
          <div>Тикет #{issueId} создан</div>
          <div>
            Посмотреть можно по <Link to={getIssueLink(issueId)}>ссылке</Link>
          </div>
        </div>
      );
    }

    return (
      <CompleteOverlay isComplete={this.props.dialog.status} message={message}>
        <div className={css.header}>
          <Title className={css.title}>{title}</Title>
          <div className={css.requiredInfo}>
            Поля обозначенные <span className={css.requiredStar}>*</span> &#8212; обязательные
          </div>
        </div>
        <div className={css.content}>
          {categoryRender()}
          {clientRender()}
          {renderPriority()}
          <AttachFiles
            files={this.props.dialog.data.files}
            onDrop={this._handleFilesAdd}
            onFilesUpdate={this.onFilesUpdate}
            onRemove={this._handleFileRemove}
          />
          {!error && this._renderForm()}
        </div>
        <TreeView onChange={onChangeCategory} />
        {error && (
          <div className={css.footer}>
            <span>Заявка сохранена не полностью, нажмите &quot;Сохранить&quot; ещё раз</span>
            <Button size="m" view="danger" onClick={this.handleSubmit}>
              Сохранить
            </Button>
          </div>
        )}
      </CompleteOverlay>
    );
  }
}
