import React from 'react';
import cx from 'classnames';
import { Tabs, Tab, ReactTabElement } from 'lego/components/Tabs';
import Comment from 'components/Comment';
import { PureChatList } from 'modules/chat';
import { CommentType } from 'components/Comment/Comment';
import { IMessage } from 'modules/chat/components/ChatList/types';
import T from 'constants/backendTypesRaw';
import Mail from './Mail';
import css from './styles.modules.scss';

interface Props {
  communication: {
    type: string;
    mail?: unknown;
    clientMail?: unknown;
    requestId?: number | string;
    ticketId?: number | string;
    issueId?: number;
    url?: string;
    chatChunk: {
      items: IMessage[];
    };
    comment: CommentType;
    contentHtml?: string;
  };
  className: string;
}

const Communication = (props: Props) => {
  const { communication, className } = props;

  if (!communication) {
    return <div className={cx(className, css.communication__link_noTabs)}>Коммуникации нет</div>;
  }

  const tabs: ReactTabElement[] = [];
  const addCommonTabFunc = (innerHtml) => {
    tabs.push(
      <Tab title="Основная информация" key="INFO">
        {innerHtml}
      </Tab>,
    );
  };
  switch (communication.type) {
    case T.TYPES.MAIL:
      tabs.push(
        <Tab title="Ответ менеджера" key="MANAGER_MAIL">
          <Mail {...communication.mail} />
        </Tab>,
      );

      if (communication.clientMail) {
        tabs.push(
          <Tab title="Письмо клиента" key="CLIENT_MAIL">
            <Mail {...communication.clientMail} />
          </Tab>,
        );
      }
      break;
    case T.TYPES.REQ_REQUEST:
      addCommonTabFunc(
        <div className={cx(className, css.communication__link_noTabs)}>
          <a
            target="_blank"
            rel="noopener noreferrer"
            href={`#/request/${communication.requestId}`}
          >
            Ссылка на заявку
          </a>
        </div>,
      );
      break;
    case T.TYPES.SP_TICKET:
      addCommonTabFunc(
        <div className={cx(className, css.communication__link_noTabs)}>
          <a target="_blank" rel="noopener noreferrer" href={`#/ticket/${communication.ticketId}`}>
            Ссылка на тикет
          </a>
        </div>,
      );
      break;
    case T.TYPES.ISSUE:
      addCommonTabFunc(
        <div className={cx(className, css.communication__link_noTabs)}>
          <a target="_blank" rel="noopener noreferrer" href={`#/cases/${communication.issueId}`}>
            Ссылка на тикет
          </a>
        </div>,
      );
      break;
    case T.TYPES.SP_CALL:
      addCommonTabFunc(
        <div className={cx(className, css.communication__link_noTabs)}>
          Звонок можно прослушать по ссылке:&nbsp;
          <a target="_blank" rel="noopener noreferrer" href={communication.url}>
            прослушать
          </a>
        </div>,
      );
      break;
    case T.TYPES.YC_CALL:
      addCommonTabFunc(
        <div className={cx(className, css.communication__link_noTabs)}>
          Звонок можно прослушать по ссылке:&nbsp;
          <a target="_blank" rel="noopener noreferrer" href={communication.url}>
            прослушать
          </a>
        </div>,
      );
      break;
    case T.TYPES.OCOMMENT:
    case T.TYPES.REQUEST_MESSAGE:
      addCommonTabFunc(
        <div className={cx(className, css.communication__link_noTabs)}>
          <Comment comment={communication.comment} />
        </div>,
      );
      break;
    case T.TYPES.CHAT_CHUNK:
      addCommonTabFunc(
        <div className={cx(className, css.chatChunkWrap)}>
          <PureChatList
            messages={props.communication.chatChunk.items}
            className={css.communicationChatChunk}
          />
        </div>,
      );
      break;
    default:
      addCommonTabFunc(
        <div className={cx(className, css.communication__link_noTabs)}>
          Wrong type communication
        </div>,
      );
  }

  if (communication.contentHtml)
    tabs.push(
      <Tab title="Дополнительно" key="ADDITIONAL">
        <span dangerouslySetInnerHTML={{ __html: communication.contentHtml }} />
      </Tab>,
    );

  return (
    <div className={className}>
      <Tabs navigationTheme="start" className={css.tabsWrap}>
        {tabs}
      </Tabs>
    </div>
  );
};

export default Communication;
