import React from 'react';
import get from 'lodash/get';
import Button from '@crm/components/dist/lego2/Button';
import Toolbar from 'components/Toolbar';
import StaffCard from 'components/StaffCard';
import TreeInput from 'lego/components/TreeInput';
import css from '../styles.modules.scss';
import ClientReview from './ClientReview';
import ReasonChange from './ReasonChange';
import Conflict from './Conflict';
import Calculator from '../Calculator';

const TYPES = {
  DEFAULT: 'DEFAULT',
  CLIENT: 'CLIENT',
  CROSS: 'CROSS',
};

export default class Review extends React.Component {
  static reviewerCanItemClick = (entity) => entity.type === 1;
  static reviewerGetSlugId = (entity) => `${entity.type}_${entity.id}`;

  static defaultProps = {
    conflictScore: 0,
    comment: '',
  };

  constructor(props) {
    super(props);

    this.state = {
      isEdit: this._isEdit(),
      isChange: false,
      isValid: true,
      isCalcValid: true,
      canSkip: false,
      ...this.getInitState(),
    };
  }

  componentDidMount() {
    // eslint-disable-next-line react/no-did-mount-set-state
    let calcValidation = this._calc && this._calc.isValid();
    calcValidation = typeof calcValidation === 'boolean' ? calcValidation : true;
    this.setState({
      isCalcValid: calcValidation,
      isValid: this.isValid(),
    });
  }

  getInitState(props) {
    const { conflictScore, comment } = props || this.props;

    return {
      conflict: conflictScore,
      reason: '0',
      customReason: '',
      comment,
    };
  }

  handleChangeInput = (e) => {
    const { name, value } = e.target;
    if (name) {
      this.setState({ [name]: value }, this._updateValid);
    }
  };

  isValid() {
    const { isChange, reason, customReason } = this.state;
    let isValid = true;
    if (isChange) {
      isValid =
        (reason !== '0' && reason !== 'OTHER') || (reason === 'OTHER' && customReason.length > 0);
    }

    return !this.state.isEdit || (isValid && this.state.isCalcValid);
  }

  _updateValid = () => {
    const { comment } = this.state;

    this.setState({
      isValid: this.isValid(),
      canSkip: comment.length > 0,
    });
  };

  _isEdit() {
    if (this.props.type === TYPES.CLIENT) {
      return !(this.props.readonly || this.props.skipped);
    }

    return !(this.props.readonly || this.props.skipped || this.props.score !== undefined);
  }

  _handleCalcChange = () => {
    let isCalcValid = true;
    if (this._calc) {
      isCalcValid = this._calc && this._calc.isValid();
    }
    if (isCalcValid !== this.state.isCalcValid) {
      this.setState({
        isCalcValid,
      });
      this._updateValid();
    }
  };

  _handleChange = () => {
    this.setState({
      isEdit: true,
      isChange: true,
      isValid: false,
    });
  };

  _handleCancel = () => {
    this.setState({
      isEdit: false,
      isChange: false,
      ...this.getInitState(),
    });

    this._calc && this._calc.reset();
  };

  _getCalcVersion = () => get(this.props, 'calc.id');

  _handleSave = () => {
    const { isChange, reason, customReason, conflict, comment } = this.state;

    let reasonForChange;
    if (isChange) {
      reasonForChange = reason !== 'OTHER' ? reason : customReason;
    }

    const score = this._calc && this._calc.value();
    this.props
      .saveItem({
        conflictScore: conflict,
        comment,
        score: score != null ? score : undefined,
        skip: this.props.type === Review.TYPES.CLIENT || undefined,
        reasonForChange,
        calcId: this._getCalcVersion(),
        role: this.props.tab || undefined,
        calcData: (this._calc && JSON.stringify(this._calc.getData())) || undefined,
      })
      .then(() => {
        this.setState({
          isEdit: false,
          isChange: false,
          ...this.getInitState({ conflictScore: conflict, comment }),
        });
      });
  };

  _skip = (message = '', stopNextReview = false) => {
    if (!confirm(message)) {
      return;
    }

    const { conflict, comment } = this.state;

    this.props
      .saveItem({
        role: this.props.tab || undefined,
        conflictScore: this.props.tab === 'client' ? conflict : undefined,
        skip: true,
        stopNextReview,
        comment,
      })
      .then(() => {
        this.setState({
          isEdit: false,
          isChange: false,
        });
      });
  };

  _handleSkip = () => {
    this._skip('Действие нельзя отменить. Подтвердить?');
  };

  _handleHardSkip = () => {
    this._skip('Удалить данную коммуникацию без оценки?', true);
  };

  _handleReviewerChange = (reviewer) => {
    if (!reviewer) {
      return;
    }

    this.props.saveReviewer({ reviewerId: reviewer.id }).then(() => {
      this.setState({ isEdit: false, isChange: false });
    });
  };

  renderCalc() {
    let dataCalc;
    try {
      dataCalc = JSON.parse(get(this.props, 'calc.data'));
    } catch (e) {}

    let meta;
    try {
      meta = JSON.parse(this.props.calc.meta);
    } catch (e) {
      return null;
    }

    const versionId = this._getCalcVersion();

    if (!meta) {
      return null;
    }

    const calcProps = {
      onChange: this._handleCalcChange,
      id: versionId,
      score: get(this.props, 'score'),
      data: dataCalc,
      ref: (node) => {
        this._calc = node;
      },
      isEdit: this.state.isEdit,
    };

    return React.createElement(Calculator, {
      ...calcProps,
      disabled: !this.state.isEdit,
      meta,
    });
  }

  renderComment() {
    if (this.state.isEdit) {
      return (
        <div>
          <div className={css.label}>Комментарий:</div>
          <textarea
            placeholder="Комментарий"
            name="comment"
            value={this.state.comment}
            onChange={this.handleChangeInput}
            className={css.comment}
            cols="30"
            rows="5"
          />
        </div>
      );
    }
    return (
      <div className={css.commentPreview}>
        <div className={css.label}>Комментарий:</div>
        <pre>{this.props.comment}</pre>
      </div>
    );
  }

  renderButtons() {
    if (this.props.readonly || this.props.skipped) {
      return null;
    }

    const buttons = [];

    if (this.state.isChange) {
      const saveProps = {
        view: 'action',
        onClick: this._handleSave,
        disabled: !this.state.isValid,
        children: 'Сохранить',
      };
      buttons.push(<Button key="Save" {...saveProps} />);

      const cancelProps = {
        onClick: this._handleCancel,
        children: 'Отменить',
      };
      buttons.push(<Button key="Cancel" {...cancelProps} />);
    } else if (this.state.isEdit) {
      if (this.props.type === Review.TYPES.CLIENT) {
        const ignoreProps = {
          view: 'action',
          onClick: this._handleSkip,
          disabled: !this.state.isValid,
          children: 'Не учитывать',
        };
        buttons.push(<Button key="Ignore" {...ignoreProps} />);
      } else {
        const saveProps = {
          view: 'action',
          onClick: this._handleSave,
          disabled: !this.state.isValid,
          children: 'Сохранить',
        };
        buttons.push(<Button key="Save" {...saveProps} />);

        if (this.props.type !== Review.TYPES.CROSS) {
          const skipProps = {
            onClick: this._handleSkip,
            disabled: !this.state.canSkip,
            children: 'Пропустить',
          };
          buttons.push(<Button key="Skip" {...skipProps} />);
        }

        const hardSkipProps = {
          onClick: this._handleHardSkip,
          disabled: !this.state.canSkip,
          children: 'Не оценивать',
        };
        buttons.push(<Button key="Hard skip" {...hardSkipProps} />);
      }
    } else {
      const editProps = {
        view: 'action',
        onClick: this._handleChange,
        children: 'Изменить',
      };
      buttons.push(<Button key="Edit" {...editProps} />);
    }

    return <Toolbar className={css.buttons}>{buttons}</Toolbar>;
  }

  render() {
    const { readonly, skipped } = this.props;
    const { isChange, isEdit, reason, customReason, conflict } = this.state;
    return (
      <div>
        <div className={css.reviewer}>
          <StaffCard
            user={this.props.reviewer}
            theme={StaffCard.THEME.YANDEX}
            className={css['reviewer-staffcard']}
          />
          {!(readonly || skipped) && (
            <TreeInput
              onChange={this._handleReviewerChange}
              canItemClick={Review.reviewerCanItemClick}
              getSlugId={Review.reviewerGetSlugId}
              endpoint="/view/staff/user/tree?part=all"
              placeholder="Изменить"
              destroyOnUnmount
            />
          )}
        </div>
        {this.props.type !== TYPES.CLIENT && this.renderCalc()}
        {this.props.type === TYPES.CLIENT && (
          <ClientReview comment={this.props.clientComment} reason={this.props.clientReason} />
        )}
        <Conflict value={conflict} disabled={!isEdit} onChange={this.handleChangeInput} />
        {this.renderComment()}
        {isChange && (
          <ReasonChange
            reason={reason}
            customReason={customReason}
            onChange={this.handleChangeInput}
          />
        )}
        {this.renderButtons()}
      </div>
    );
  }
}

Review.TYPES = TYPES;
