import React from 'react';
import get from 'lodash/get';
import { Tabs, Tab, ReactTabElement } from 'lego/components/Tabs';
import css from './styles.modules.scss';
import Review from './Review';
import History from './History';
import Communication from './Communication';

const TAB_ROLE = {
  '0': 'teamlead',
  '1': 'client',
  '2': 'crosslead',
};

interface Props {
  item: any;
  saveItem: (args: any) => {};
  saveReviewer: (args: any) => {};
}

class Item extends React.Component<Props> {
  private handleSave = args =>
    this.props.saveItem({
      managerId: this.props.item.manager.id,
      id: this.props.item.id,
      ...args,
    });

  private handleReviewerSave = args =>
    this.props.saveReviewer({
      managerId: this.props.item.manager.id,
      id: this.props.item.id,
      ...args,
    });

  private renderCalcTab() {
    const tabs: ReactTabElement[] = [];
    let index = -1;
    let isDefaultIndex = false;

    if (get(this.props.item, 'permissions.teamlead.visible')) {
      index += 1;
      if (!isDefaultIndex && get(this.props, 'ui.tabIndex') === index) {
        isDefaultIndex = true;
      }

      tabs.push(
        <Tab
          key={`TEAMLEAD_${this.props.item.id || 'new'}`}
          title={this.renderLabel('teamlead')}
          className={css.calc}
          keepMount
        >
          <Review
            tab={TAB_ROLE[0]}
            {...this.props.item.teamleadReview}
            saveItem={this.handleSave}
            saveReviewer={this.handleReviewerSave}
            reviewType={this.props.item.reviewType}
            reviewer={this.props.item.reviewer}
            readonly={!get(this.props.item, 'permissions.teamlead.canEdit')}
          />
        </Tab>,
      );
    }

    if (get(this.props.item, 'permissions.client.visible')) {
      index += 1;
      if (!isDefaultIndex && get(this.props, 'ui.tabIndex') === index) {
        isDefaultIndex = true;
      }

      tabs.push(
        <Tab
          key={`CLIENT_${this.props.item.id || 'new'}`}
          title={this.renderLabel('client')}
          className={css.calc}
          keepMount
        >
          <Review
            tab={TAB_ROLE[1]}
            type={Review.TYPES.CLIENT}
            {...this.props.item.clientReview}
            saveItem={this.handleSave}
            saveReviewer={this.handleReviewerSave}
            reviewType={this.props.item.reviewType}
            reviewer={this.props.item.reviewer}
            readonly
          />
        </Tab>,
      );
    }

    if (get(this.props.item, 'permissions.crosslead.visible')) {
      index += 1;
      if (!isDefaultIndex && get(this.props, 'ui.tabIndex') === index) {
        isDefaultIndex = true;
      }

      tabs.push(
        <Tab
          key={`CROSSLEAD_${this.props.item.id || 'new'}`}
          title={this.renderLabel('crosslead')}
          className={css.calc}
          keepMount
        >
          <Review
            tab={TAB_ROLE[2]}
            type={Review.TYPES.CROSS}
            {...this.props.item.crossleadReview}
            saveItem={this.handleSave}
            saveReviewer={this.handleReviewerSave}
            reviewType={this.props.item.reviewType}
            reviewer={this.props.item.reviewer}
            readonly={!get(this.props.item, 'permissions.crosslead.canEdit')}
          />
        </Tab>,
      );
    }

    if (this.props.item.scoreHistory && this.props.item.scoreHistory.length) {
      tabs.push(
        <Tab title="История изменения оценки" key="HISTORY" className={css.calc} keepMount>
          <History items={this.props.item.scoreHistory} />
        </Tab>,
      );
    }

    return (
      <Tabs defaultValue={(isDefaultIndex && index) || undefined} navigationTheme="start" canHide>
        {tabs}
      </Tabs>
    );
  }

  private renderLabel(type) {
    const TITLE = {
      teamlead: 'Внутренняя оценка',
      crosslead: 'Кросс-оценка',
      client: 'Оценка клиента',
    };

    const CLASS_NAME = {
      teamlead: css.score_teamlead,
      crosslead: css.score_crosslead,
      client: css.score_client,
    };

    const title = TITLE[type];
    const score = get(this.props, `item.${type}Review.score`);
    const className = CLASS_NAME[type];

    const crossLead = (
      <span>
        {title}&nbsp;
        {score !== undefined ? <span className={className}>{score}</span> : null}
      </span>
    );

    return crossLead;
  }

  public render() {
    return (
      <div className={css.content}>
        {this.props.item && this.props.item.isLoad && (
          <React.Fragment>
            <div className={css.review}>{this.renderCalcTab()}</div>
            <Communication className={css.communication} communication={this.props.item.item} />
          </React.Fragment>
        )}
      </div>
    );
  }
}

export default Item;
