import React from 'react';
import PropTypes from 'prop-types';
import Empty from 'components/Empty';
import { Tabs, Tab } from 'lego/components/Tabs';
import get from 'lodash/get';
import { MODULE_NAME } from '../../constants';
import css from './styles.modules.scss';
import tableCss from './table.modules.scss';
import Manager from './Manager';
import ReviewFilterContainer from './ReviewFilterContainer';

const renderEmpty = () => <Empty className={tableCss.empty} text="Оценок нет" icon={null} />;

const header = (
  <div className={tableCss.header}>
    <div className={tableCss.date}>Дата</div>
    <div className={tableCss.type}>Тип</div>
    <div className={tableCss.mark}>Оценка</div>
  </div>
);

export default class Managers extends React.Component {
  static propTypes = {
    managers: PropTypes.instanceOf(Object),
    ui: PropTypes.instanceOf(Object).isRequired,
    onAddItem: PropTypes.func.isRequired,
    onTabClick: PropTypes.func.isRequired,
    onFilterUpdate: PropTypes.func.isRequired,
    groupIds: PropTypes.number,
  };

  static defaultProps = {
    managers: {},
    groupIds: undefined,
  };

  handleAddClick = (e, args) => {
    e.preventDefault();
    e.stopPropagation();

    // eslint-disable-next-line no-restricted-globals
    if (confirm('Вы точно хотите добавить новую активность для менеджера?')) {
      this.props.onAddItem(args);
    }
  };

  render() {
    if (!this.props.managers) {
      return null;
    }

    const { groupIds } = this.props;

    const selectedGroups = groupIds && groupIds.length > 0 ? groupIds : [];

    const tabs = [];

    if (get(this.props.managers, 'permission.tab.teamlead')) {
      const isRenderTeamlead = !!get(this.props, 'managers.teamlead.length');
      const canAddItem = !!get(this.props, 'managers.permission.canAddItem');

      tabs.push(
        <Tab className={css.tab} title="Внутренние оценки" key="TEAMLEAD" value="teamlead">
          <ReviewFilterContainer
            items={this.props.managers.groups}
            onSubmit={this.props.onFilterUpdate}
            selectedGroups={selectedGroups}
            enableReinitialize
          />
          {isRenderTeamlead && header}
          <div className={tableCss.root}>
            <div className={tableCss.body}>
              {isRenderTeamlead && (
                <div className={tableCss.content}>
                  {this.props.managers.teamlead.map(group => (
                    <div key={group.groupName}>
                      <div className={tableCss.group_title} title={group.groupName}>
                        {group.groupName}
                      </div>
                      {group.reviews.map(manager => (
                        <Manager
                          key={manager.manager.id}
                          manager={manager}
                          addCommunication={this.handleAddClick}
                          isAddButton={canAddItem}
                          isTeamleadMark
                        />
                      ))}
                    </div>
                  ))}
                </div>
              )}
              {!isRenderTeamlead && renderEmpty()}
            </div>
          </div>
        </Tab>,
      );
    }

    if (get(this.props.managers, 'permission.tab.client')) {
      const isRenderClient = !!get(this.props, 'managers.client.length');

      tabs.push(
        <Tab className={css.tab} title="Внешние оценки" key="CLIENT" value="client">
          <ReviewFilterContainer
            items={this.props.managers.groups}
            onSubmit={this.props.onFilterUpdate}
            selectedGroups={selectedGroups}
            enableReinitialize
          />
          {isRenderClient && header}
          <div className={tableCss.root}>
            <div className={tableCss.body}>
              {isRenderClient && (
                <div className={tableCss.content}>
                  {this.props.managers.client.map(group => (
                    <div key={group.groupName}>
                      <div className={tableCss.group_title}>{group.groupName}</div>
                      {group.reviews.map(manager => (
                        <Manager
                          key={manager.manager.id}
                          manager={manager}
                          addCommunication={this.handleAddClick}
                        />
                      ))}
                    </div>
                  ))}
                </div>
              )}
              {!isRenderClient && renderEmpty()}
            </div>
          </div>
        </Tab>,
      );
    }

    return (
      <Tabs
        className={css.root}
        classNameHeader={css.header}
        value={this.props.ui.tabName}
        navigationTheme="start"
        onChange={this.props.onTabClick}
        headerLeft={<div className={css.title}>{MODULE_NAME}</div>}
        showHeader
      >
        {tabs}
      </Tabs>
    );
  }
}
