import { createElement, Component } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import get from 'lodash/get';
import globalActions from 'actions';
import { STATE_NAME } from '../constants';
import * as reviewActions from '../actions';
import reducers from '../reducers';
import ReviewLayout from '../components/Layout';
import { getReviewFilterGroupValue, getReviewFilterNotVerifiedValue } from '../selectors';

global.reduxStore.injectReducer(STATE_NAME, reducers);

const mapState = (state) => ({
  list: get(state, 'review.list'),
  item: get(state, 'review.item'),
  ui: get(state, 'review.ui'),
  groupIds: getReviewFilterGroupValue(state),
  onlyNotVerified: getReviewFilterNotVerifiedValue(state),
});

const mapDispatch = (dispatch) => ({
  fetch: (args) => dispatch(reviewActions.getList(args)),
  getItem: (args) => dispatch(reviewActions.getItem(args)),
  resetItem: () => dispatch(reviewActions.resetItem),
  saveItem: (args) => {
    const { managerId, ...other } = args;
    return dispatch(reviewActions.saveItem(other));
  },
  saveReviewer: (args) => dispatch(reviewActions.saveReviewer(args)),
  setTabIndex: (name) => dispatch(reviewActions.setPanel(name)),
  add: (args) => dispatch(reviewActions.addItem(args)),
  dispatch,
});

const mergeMap = (stateProps, dispatchProps, ownProps) => {
  const update = (args) => {
    return (data) => {
      dispatchProps.getItem({
        id: args.id,
        role: args.role === 'client' ? 'teamlead' : args.role,
      });
      dispatchProps.fetch({
        type: stateProps.ui.tabName,
        groupIds: stateProps.groupIds,
        onlyNotVerified: stateProps.onlyNotVerified,
      });
      if (data.unableAddItem) {
        dispatchProps.dispatch(globalActions.uiToggleSnackbar(true, `${data.message}`, 'ERROR'));
      }
      return null;
    };
  };

  return {
    ...ownProps,
    ...stateProps,
    ...dispatchProps,
    getItem: (args) => dispatchProps.getItem({ ...args }),
    setTabIndex: (name) => {
      dispatchProps
        .fetch({
          type: name,
          groupIds: stateProps.groupIds,
          onlyNotVerified: stateProps.onlyNotVerified,
        })
        .then(() => {
          dispatchProps.setTabIndex(name);
        });
    },
    add: (args) =>
      dispatchProps.add(args).then((data) => {
        if (data.unableAddItem) {
          dispatchProps.dispatch(globalActions.uiToggleSnackbar(true, `${data.message}`, 'ERROR'));
        } else {
          dispatchProps.fetch({
            type: stateProps.ui.tabName,
            groupIds: stateProps.groupIds,
            onlyNotVerified: stateProps.onlyNotVerified,
          });
        }
      }),

    saveItem: (args) => {
      const { managerId, ...other } = args;
      return dispatchProps.saveItem(other).then(update(args));
    },

    saveReviewer: (args) => {
      const { managerId, ...other } = args;
      return dispatchProps.saveReviewer(other).then(update(args));
    },
  };
};

const INTERVAL_DELAY = 60000;

@connect(mapState, mapDispatch, mergeMap)
class ReviewPage extends Component {
  static propTypes = {
    params: PropTypes.instanceOf(Object),
    ui: PropTypes.instanceOf(Object),
    getItem: PropTypes.func.isRequired,
    resetItem: PropTypes.func.isRequired,
    fetch: PropTypes.func.isRequired,
    groupIds: PropTypes.number,
    onlyNotVerified: PropTypes.bool,
  };

  static defaultProps = {
    params: undefined,
    ui: undefined,
    groupIds: undefined,
    onlyNotVerified: true,
  };

  constructor(props) {
    super(props);

    this.update();
    this.interval = setInterval(this.update, INTERVAL_DELAY);
  }

  componentDidMount() {
    if (this.props.params && this.props.params.id) {
      this.props.getItem({ id: this.props.params.id });
    }
  }

  componentWillReceiveProps(nextProps) {
    const newItemId = get(nextProps, 'params.id');
    const currentItemId = get(this.props, 'params.id');

    if (newItemId !== currentItemId) {
      if (newItemId) {
        this.props.getItem({ id: newItemId });
      } else {
        this.props.resetItem();
      }
    }
  }

  componentWillUnmount() {
    clearInterval(this.interval);
  }

  update = () => {
    this.props.fetch({
      type: this.props.ui.tabName,
      groupIds: this.props.groupIds,
      onlyNotVerified: this.props.onlyNotVerified,
    });
  };

  render() {
    return createElement(ReviewLayout, {
      ...this.props,
      onFilterUpdate: () =>
        this.props.fetch({
          type: this.props.ui.tabName,
          groupIds: this.props.groupIds,
          onlyNotVerified: this.props.onlyNotVerified,
        }),
    });
  }
}

export default ReviewPage;
