import React, { FC, useEffect, useState } from 'react';
import { Switch, Route as DomRoute, RouteComponentProps, useHistory } from 'react-router-dom';
import { AlertDTO } from 'types/dto/AlertDTO';
import { push } from 'modules/notificationsUI';
import { useDispatch } from 'react-redux';
import { Layout } from '../Layout';
import { EditForm } from '../EditForm';
import { NewForm } from '../NewForm';
import { fetchAlerts, findAlertById, saveValues, dateDescSortComparer } from './Route.utils';
import { Values } from '../Form';
import { AlertsTableAside } from './AlertsTablesAside';

export const Route: FC<RouteComponentProps> = (props) => {
  const { match } = props;
  const { path } = match;
  const dispatch = useDispatch();
  const history = useHistory();
  const [alerts, setAlerts] = useState<AlertDTO[]>([]);

  useEffect(() => {
    fetchAlerts().then((alerts = []) => {
      setAlerts(alerts);
    });
  }, []);

  const handleAlertClick = (alertId: number) => {
    history.push(`${path}/${alertId}`);
  };

  const handleSubmit = (values: Values) => {
    saveValues({
      ...values,
      categoryId: (values.category || {}).id,
    }).then((response) => {
      const restAlerts = alerts.filter((alert) => alert.id !== response.id);
      const newAlerts = [
        ...restAlerts,
        {
          ...response,
        },
      ];
      setAlerts(
        newAlerts.sort(dateDescSortComparer<AlertDTO>((alert) => new Date(alert.expiredOn))),
      );
      dispatch(
        push({
          level: 'success',
          title: 'Алерт',
          text: 'Алерт сохранен',
        }),
      );
      history.push(`${path}/${response.id}`);
    });
  };

  return (
    <Switch>
      <DomRoute
        path={`${match.url}/new`}
        render={() => (
          <Layout
            {...props}
            asideAppend={<AlertsTableAside alerts={alerts} onAlertClick={handleAlertClick} />}
          >
            <NewForm onSubmit={handleSubmit} />
          </Layout>
        )}
      />
      <DomRoute
        path={`${match.url}/:id?`}
        render={(renderProps) => (
          <Layout
            {...props}
            asideAppend={
              <AlertsTableAside
                alerts={alerts}
                onAlertClick={handleAlertClick}
                activeAlertId={Number(renderProps.match.params.id)}
              />
            }
          >
            <EditForm
              alert={findAlertById(alerts, renderProps.match.params.id)}
              onSubmit={handleSubmit}
            />
          </Layout>
        )}
      />
    </Switch>
  );
};
