import { get, post, put } from 'api/common';
import { AlertDTO } from 'types/dto/AlertDTO';
import { types } from '../../../../components/TitleTable';
import { SaveValues } from '../Form/Form.types';

export const fetchAlerts = (): Promise<AlertDTO[]> => {
  return get({
    url: '/alerts',
    data: {
      offset: 0,
      length: 1000,
      sortType: 'desc',
      sortField: 'expiredOn',
    },
  }).then((response) => response.items);
};

export const dateDescSortComparer = <T extends {}>(getter: (el: T) => Date) => {
  return (a, b) => getter(b).valueOf() - getter(a).valueOf();
};

export const filterActiveAlerts = (alerts: AlertDTO[]): AlertDTO[] => {
  return alerts.filter((alert) => new Date() < new Date(alert.expiredOn));
};

export const filterInactiveAlerts = (alerts: AlertDTO[]): AlertDTO[] => {
  return alerts.filter((alert) => new Date() > new Date(alert.expiredOn));
};

export const getCaptionForISOString = (ISOString: string): string => {
  const date = new Date(ISOString);
  return `${date.toLocaleDateString('ru')}, ${date.toLocaleTimeString('ru', {
    hour: '2-digit',
    minute: '2-digit',
  })}`;
};

export const schemeFromAlerts = (alerts: AlertDTO[], activeAlertId?: number): types.TableScheme => {
  return {
    columns: [
      {
        caption: 'Название',
      },
      {
        caption: 'Категория',
      },
      {
        caption: 'Активен до',
      },
    ],
    rows: alerts.map((alert) => ({
      id: alert.id,
      cells: [
        {
          caption: alert.name,
        },
        {
          caption: alert.category.name,
        },
        {
          caption: getCaptionForISOString(alert.expiredOn),
        },
      ],
    })),
    activeRows: activeAlertId
      ? {
          byId: {
            [activeAlertId]: true,
          },
        }
      : undefined,
  };
};

export const findAlertById = (
  alerts: AlertDTO[],
  rawId: number | string | undefined,
): AlertDTO | undefined => {
  if (rawId == null) {
    return undefined;
  }

  const id = Number(rawId);
  return alerts.find((alert) => alert.id === id);
};

export const saveValues = (values: SaveValues): Promise<AlertDTO> => {
  if (values.id != null) {
    return put({
      url: '/alerts',
      data: values,
    });
  }

  return post({
    url: '/alerts',
    data: values,
  });
};
