import React from 'react';
import PropTypes from 'prop-types';
import RouterLink from 'components/RouterLink';
import Empty from 'components/Empty';
import cx from 'classnames';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import css from './styles.modules.scss';
import * as keyset from './DelegationsList.i18n';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nNoDelegations = i18n('noDelegations');
const i18nActive = i18n('active');
const i18nNonActive = i18n('nonActive');
const i18nFromWhom = commonI18n('fromWhom');
const i18nToWhom = commonI18n('toWhom');
const i18nFrom = commonI18n('from');
const i18nTo = commonI18n('to');

const RenderItem = ({ item, formatDate, editable }) => {
  const isEditableClass = editable ? css.isEditable : css.isReadOnly;

  return editable ? (
    <RouterLink
      className={cx(css.listItem, isEditableClass)}
      activeClassName={css.listItemCurrent}
      to={`/settings/delegations/${item.delegationId}`}
    >
      <span className={cx(css.col, css.name)}>
        <span className={css.userName} title={item.userName}>
          {item.userName}
        </span>
      </span>
      <span className={cx(css.col, css.name)}>
        <span className={css.userName} title={item.executorName}>
          {item.executorName}
        </span>
      </span>
      <span className={cx(css.col, css.date)}>{formatDate(item.from)}</span>
      <span className={cx(css.col, css.date)}>{formatDate(item.to)}</span>
    </RouterLink>
  ) : (
    <span key={item.delegationId} className={cx(css.listItem, isEditableClass)}>
      <span className={cx(css.col, css.name)}>
        <span className={css.userName} title={item.userName}>
          {item.userName}
        </span>
      </span>
      <span className={cx(css.col, css.name)}>
        <span className={css.userName} title={item.executorName}>
          {item.executorName}
        </span>
      </span>
      <span className={cx(css.col, css.date)}>{formatDate(item.from)}</span>
      <span className={cx(css.col, css.date)}>{formatDate(item.to)}</span>
    </span>
  );
};

const SettingsDelegaitionsList = props => {
  const { className, items: delegations, formatDate } = props;
  const activeDelegations = [];
  const inactiveDelegations = [];
  delegations.forEach(item => {
    const editable = item.canEditStartDate || item.canEditEndDate;
    const element = (
      <RenderItem key={item.delegationId} item={item} formatDate={formatDate} editable={editable} />
    );

    if (editable) {
      activeDelegations.push(element);
    } else {
      inactiveDelegations.push(element);
    }
  });

  if (!delegations.length) {
    return <Empty text={i18nNoDelegations} icon={null} />;
  }

  return (
    <div className={className}>
      <div className={css.data}>
        {[
          { title: i18nActive, delegations: activeDelegations },
          { title: i18nNonActive, delegations: inactiveDelegations },
        ].map(group => {
          if (!Array.isArray(group.delegations) || !group.delegations.length) {
            return null;
          }

          return (
            <div className={css.delegationGroup}>
              <div className={css.header}>
                <div className={css.delegationGroupHeader}>{group.title}</div>
                <div className={css.content}>
                  <div className={cx(css.col, css.name)}>{i18nFromWhom}</div>
                  <div className={cx(css.col, css.name)}>{i18nToWhom}</div>
                  <div className={cx(css.col, css.date)}>{i18nFrom}</div>
                  <div className={cx(css.col, css.date)}>{i18nTo}</div>
                </div>
              </div>
              <div className={css.list}>{group.delegations}</div>
            </div>
          );
        })}
      </div>
    </div>
  );
};

SettingsDelegaitionsList.propTypes = {
  className: PropTypes.string,
  items: PropTypes.arrayOf(PropTypes.object),
  formatDate: PropTypes.func,
};

SettingsDelegaitionsList.defaultProps = {
  className: undefined,
  items: [],
  formatDate: () => {},
};

export default SettingsDelegaitionsList;
